<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ========= صلاحيات (هوك بسيط) =========
   عدّل المنطق ده حسب نظامك:
   مثال: $canEdit = ($_SESSION['role'] ?? '') === 'admin';
*/
$canEdit = true;

/* ========= CSRF ========= */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$csrf = $_SESSION['csrf_token'];

/* ========= Helpers ========= */
function clean_str(?string $s, int $max = 255): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;
  return mb_substr($s, 0, $max);
}
function slugify(?string $s): ?string {
  $s = trim((string)$s);
  if ($s === '') return null;
  $s = mb_strtolower($s, 'UTF-8');
  $s = preg_replace('~[^\p{Arabic}a-z0-9\s\-]+~u', '', $s);
  $s = preg_replace('~[\s\-]+~u', '-', $s);
  $s = trim($s, '-');
  return $s ?: null;
}
function unique_slug(PDO $pdo, string $base, ?int $ignoreId=null): string {
  $slug = $base ?: ('cat-'.time());
  $i=2;
  while (true) {
    $sql = "SELECT 1 FROM categories WHERE slug = :s".
           ($ignoreId ? " AND id <> :id" : "").
           " LIMIT 1";
    $st = $pdo->prepare($sql);
    $params = [':s'=>$slug];
    if ($ignoreId) $params[':id']=$ignoreId;
    $st->execute($params);
    if (!$st->fetch()) return $slug;
    $slug = $base . '-' . $i++;
  }
}
function buildTree(array $rows): array {
  $byId=[]; foreach ($rows as $r){ $r['children']=[]; $byId[$r['id']]=$r; }
  $root=[];
  foreach ($byId as $id=>&$n){
    if ($n['parent_id']) {
      if (isset($byId[$n['parent_id']])) $byId[$n['parent_id']]['children'][]=&$n;
      else $root[]=&$n; // أب مفقود
    } else $root[]=&$n;
  }
  $sortFn = function(&$list) use (&$sortFn){
    usort($list, function($a,$b){
      if ((int)$a['sort_order']===(int)$b['sort_order'])
        return strnatcasecmp($a['name_ar'],$b['name_ar']);
      return (int)$a['sort_order'] <=> (int)$b['sort_order'];
    });
    foreach ($list as &$n) if (!empty($n['children'])) $sortFn($n['children']);
  };
  $sortFn($root);
  return $root;
}
function computeCounts(&$node, $directCounts) {
  $self = (int)($directCounts[$node['id']] ?? 0);
  $sum = $self;
  if (!empty($node['children'])) {
    foreach ($node['children'] as &$ch) {
      computeCounts($ch, $directCounts);
      $sum += (int)($ch['total_count'] ?? 0);
    }
  }
  $node['direct_count'] = $self;
  $node['total_count']  = $sum;
}

/* ========= Feature: Product counts if products table exists ========= */
$hasProducts = false;
try {
  $q = $pdo->query("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = 'products'");
  $hasProducts = ((int)$q->fetchColumn() > 0);
} catch (Throwable $e) { $hasProducts = false; }

$directCounts = [];
if ($hasProducts) {
  try {
    $q = $pdo->query("SELECT category_id, COUNT(*) c FROM products WHERE (deleted_at IS NULL OR deleted_at IS NULL) GROUP BY category_id");
    foreach ($q->fetchAll(PDO::FETCH_ASSOC) as $r) {
      if ($r['category_id'] !== null) $directCounts[(int)$r['category_id']] = (int)$r['c'];
    }
  } catch (Throwable $e) {
    $directCounts = []; // تجاهل لو جدول مختلف
  }
}

/* ========= Actions ========= */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF). أعد تحميل الصفحة.';
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية التعديل.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action === 'save') {
        $id        = ($_POST['id'] ?? '') !== '' ? (int)$_POST['id'] : null;
        $name_ar   = clean_str($_POST['name_ar'] ?? '', 120);
        $name_en   = clean_str($_POST['name_en'] ?? '', 120);
        $parent_id = ($_POST['parent_id'] ?? '') !== '' ? (int)$_POST['parent_id'] : null;
        $slug      = clean_str($_POST['slug'] ?? '', 150);
        $desc      = clean_str($_POST['description'] ?? '', 5000);
        $sort      = (int)($_POST['sort_order'] ?? 0);
        $active    = isset($_POST['is_active']) ? 1 : 0;

        if (!$name_ar) throw new RuntimeException('اسم التصنيف (عربي) مطلوب.');

        // التحقق من تكرار الاسم تحت نفس الأب
        $dupSql = "SELECT 1 FROM categories WHERE parent_id <=> :p AND name_ar = :n" . ($id ? " AND id<>:id" : "") . " LIMIT 1";
        $dupSt  = $pdo->prepare($dupSql);
        $dupParams = [':p'=>$parent_id, ':n'=>$name_ar];
        if ($id) $dupParams[':id']=$id;
        $dupSt->execute($dupParams);
        if ($dupSt->fetch()) throw new RuntimeException('اسم التصنيف مكرر ضمن نفس التصنيف الأب.');

        // تحضير slug فريد
        if (!$slug) $slug = slugify($name_en ?? '') ?: slugify($name_ar ?? '') ?: ('cat-'.time());
        $slug = unique_slug($pdo, $slug, $id);

        // رفع صورة (اختياري)
        $imageName = null;
        $uploadDir = __DIR__ . '/uploads/categories/';
        if (!is_dir($uploadDir)) { @mkdir($uploadDir, 0777, true); }

        if (!empty($_FILES['image']) && is_array($_FILES['image']) && ($_FILES['image']['error'] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK) {
          $allowed = ['jpg','jpeg','png','webp'];
          $ext = strtolower(pathinfo($_FILES['image']['name'], PATHINFO_EXTENSION));
          if (!in_array($ext, $allowed, true)) throw new RuntimeException('صيغة الصورة غير مسموح بها (JPG/PNG/WEBP).');
          if ((int)$_FILES['image']['size'] > 3*1024*1024) throw new RuntimeException('الحجم الأقصى 3MB.');
          $imageName = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
          if (!move_uploaded_file($_FILES['image']['tmp_name'], $uploadDir.$imageName)) {
            throw new RuntimeException('فشل رفع الصورة.');
          }
        }

        if ($id) {
          $oldImg = null;
          if ($imageName) {
            $old = $pdo->prepare("SELECT image FROM categories WHERE id=?");
            $old->execute([$id]);
            $oldImg = $old->fetchColumn() ?: null;
          }

          $sql = "UPDATE categories SET parent_id=:parent_id, name_ar=:name_ar, name_en=:name_en,
                  slug=:slug, description=:description, sort_order=:sort_order, is_active=:is_active".
                  ($imageName ? ", image=:image" : "") . " WHERE id=:id AND deleted_at IS NULL";
          $stmt = $pdo->prepare($sql);
          $params = [
            ':parent_id'=>$parent_id, ':name_ar'=>$name_ar, ':name_en'=>$name_en, ':slug'=>$slug,
            ':description'=>$desc, ':sort_order'=>$sort, ':is_active'=>$active, ':id'=>$id
          ];
          if ($imageName) $params[':image'] = $imageName;
          $stmt->execute($params);

          if ($imageName && $oldImg) {
            $oldPath = $uploadDir.$oldImg;
            if (is_file($oldPath)) @unlink($oldPath);
          }

          $success = '✔️ تم تحديث التصنيف';
        } else {
          $stmt = $pdo->prepare("INSERT INTO categories
            (parent_id, name_ar, name_en, slug, image, description, sort_order, is_active)
            VALUES (:parent_id,:name_ar,:name_en,:slug,:image,:description,:sort_order,:is_active)");
          $stmt->execute([
            ':parent_id'=>$parent_id, ':name_ar'=>$name_ar, ':name_en'=>$name_en, ':slug'=>$slug,
            ':image'=>$imageName, ':description'=>$desc, ':sort_order'=>$sort, ':is_active'=>$active
          ]);
          $success = '✔️ تم إضافة التصنيف';
        }

      } elseif ($action === 'toggle' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $pdo->prepare("UPDATE categories SET is_active = 1 - is_active WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم تغيير الحالة';

      } elseif ($action === 'soft_delete' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $pdo->prepare("UPDATE categories SET is_active=0, deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة التصنيف';

      } elseif ($action === 'restore' && isset($_POST['id'])) {
        $id = (int)$_POST['id'];
        $pdo->prepare("UPDATE categories SET deleted_at=NULL, is_active=1 WHERE id=?")->execute([$id]);
        $success = 'تم استعادة التصنيف';

      } elseif ($action === 'reorder_seq') {
        // إعادة ترتيب داخل نفس الأب (عن طريق SortableJS)
        $parent = $_POST['parent_id'] ?? 'root';
        $ids = $_POST['order'] ?? [];
        if (!is_array($ids) || empty($ids)) throw new RuntimeException('ترتيب غير صالح.');
        $order = 0;
        foreach ($ids as $cid) {
          $cid = (int)$cid;
          $st = $pdo->prepare("UPDATE categories SET sort_order=? WHERE id=?");
          $st->execute([$order++, $cid]);
        }
        $success = 'تم حفظ الترتيب';
      }

    } catch (Throwable $e) {
      $error = 'خطأ: '.htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    }
  }
}

/* ========= قراءة البيانات ========= */
$q = trim($_GET['q'] ?? '');
$showArchived = isset($_GET['show_archived']) ? 1 : 0;
$params = [];
$where = $showArchived ? "1=1" : "deleted_at IS NULL";
if ($q !== '') {
  $where .= " AND (name_ar LIKE :q OR name_en LIKE :q OR slug LIKE :q)";
  $params[':q'] = "%$q%";
}
$stmt = $pdo->prepare("SELECT * FROM categories WHERE $where ORDER BY parent_id IS NULL DESC, sort_order ASC, name_ar ASC");
$stmt->execute($params);
$rows = $stmt->fetchAll(PDO::FETCH_ASSOC);
$tree = buildTree($rows);

/* إرفاق العدادات (لو متاحة) */
if ($hasProducts && !empty($tree)) {
  // نبني نسخة مرجعية بسيطة للمرور
  $attachFn = function(&$nodes) use (&$attachFn, $directCounts){
    foreach ($nodes as &$n) {
      computeCounts($n, $directCounts);
      if (!empty($n['children'])) $attachFn($n['children']);
    }
  };
  $attachFn($tree);
}

/* بيانات لكل الـ select (الأب) من غير المؤرشفين فقط */
$allRowsStmt = $pdo->query("SELECT * FROM categories WHERE deleted_at IS NULL ORDER BY sort_order, name_ar");
$allRows = $allRowsStmt->fetchAll(PDO::FETCH_ASSOC);
$fullTree = buildTree($allRows);

function optionTreeHtml($tree, $level=0, $selected=null){
  $pad = str_repeat('— ', $level);
  $html = '';
  foreach ($tree as $n) {
    $sel = ($selected !== null && (int)$selected === (int)$n['id']) ? 'selected' : '';
    $html .= '<option value="'.$n['id'].'" '.$sel.'>'.$pad.htmlspecialchars($n['name_ar']).'</option>';
    if (!empty($n['children'])) $html .= optionTreeHtml($n['children'], $level+1, $selected);
  }
  return $html;
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>التصنيفات</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#ffffff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{
      background: linear-gradient(90deg, var(--p1), var(--p2), var(--p3), var(--p4));
      padding:14px 18px; color:#1f2a37; display:flex; align-items:center; gap:12px; justify-content:space-between;
      box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .user-info{display:flex; align-items:center; gap:10px; background:rgba(255,255,255,.6); padding:6px 10px; border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}

    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .tree ul{list-style:none;padding-left:0}
    .tree li{border:1px solid var(--border); background:var(--card); border-radius:12px; padding:10px; margin:8px 0}
    .badge-active{background:#34d399}
    .badge-inactive{background:#f59e0b}
    .img-thumb{width:40px;height:40px;object-fit:cover;border-radius:8px;border:1px solid var(--border)}
    .muted{color:var(--muted)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .sortable-placeholder{border:2px dashed var(--primary); border-radius:12px; height:56px; margin:8px 0}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-layer-group me-2"></i> التصنيفات</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <div class="card-soft">
    <div class="card-head">
      <form class="d-flex gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:260px"
               placeholder="بحث بالاسم/السلَج" value="<?= htmlspecialchars($q) ?>">
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived ? 'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> بحث</button>
        <?php if ($q || $showArchived): ?>
          <a href="categories.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>
      <?php if ($canEdit): ?>
      <div class="d-flex gap-2">
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#catModal" onclick="openCreate()">
          <i class="fa-solid fa-plus"></i> تصنيف جديد
        </button>
      </div>
      <?php endif; ?>
    </div>

    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <?php if ($q !== ''): ?>
        <div class="alert alert-info py-2">نتائج بحث عن: <strong><?= htmlspecialchars($q) ?></strong></div>
      <?php endif; ?>

      <!-- شجرة التصنيفات + سحب/إفلات -->
      <div class="tree">
        <?php
        function renderTree($nodes, $level=0, $hasProducts=false){
          if (!$nodes) { echo '<div class="muted">لا توجد تصنيفات.</div>'; return; }
          $parentAttr = $level===0 ? 'data-parent="root"' : '';
          echo '<ul class="ps-0 sortable" '.$parentAttr.'>';
          foreach ($nodes as $n){
            $badge = ((int)$n['is_active']===1) ? '<span class="badge badge-active">مفعل</span>'
                                                : '<span class="badge badge-inactive">موقوف</span>';
            $img = $n['image'] ? '<img class="img-thumb me-2" src="uploads/categories/'.htmlspecialchars($n['image']).'">' : '';
            $counts = '';
            if ($hasProducts) {
              $dc = (int)($n['direct_count'] ?? 0);
              $tc = (int)($n['total_count'] ?? $dc);
              $counts = '<span class="pill me-2"><i class="fa-solid fa-shirt"></i> '.$dc.' مباشر</span>'.
                        '<span class="pill"><i class="fa-solid fa-layer-group"></i> '.$tc.' إجمالي</span>';
            }
            echo '<li class="cat-item" data-id="'.$n['id'].'">';
            echo '<div class="d-flex align-items-center justify-content-between flex-wrap gap-2">';
            echo '<div class="d-flex align-items-center">';
            echo $img;
            echo '<div>';
            echo '<div class="fw-bold">'.$n['name_ar'].' <span class="muted">('.htmlspecialchars($n['name_en'] ?? '—').')</span></div>';
            echo '<div class="muted small">slug: <span class="pill">'.htmlspecialchars($n['slug'] ?? '—').'</span></div>';
            echo '</div>';
            echo '</div>';
            echo '<div class="d-flex align-items-center gap-2">';
            echo $counts . $badge;
            if ((int)$n['deleted_at'] !== null) {
              // لا شيء
            }
            // أزرار
            if ($GLOBALS['canEdit']) {
              if ($n['deleted_at'] === null) {
                echo '<form method="post" class="d-inline">
                        <input type="hidden" name="csrf" value="'.htmlspecialchars($_SESSION['csrf_token']).'">
                        <input type="hidden" name="action" value="toggle">
                        <input type="hidden" name="id" value="'.$n['id'].'">
                        <button class="btn btn-sm btn-outline-warning" title="تبديل الحالة"><i class="fa-solid fa-toggle-on"></i></button>
                      </form>';
                echo '<button class="btn btn-sm btn-outline-secondary" onclick=\'openEdit('.json_encode($n, JSON_UNESCAPED_UNICODE | JSON_HEX_APOS | JSON_HEX_AMP | JSON_HEX_TAG).')\' title="تعديل"><i class="fa-solid fa-pen-to-square"></i></button>';
                echo '<form method="post" class="d-inline" onsubmit="return confirm(\'أرشفة التصنيف؟\')">
                        <input type="hidden" name="csrf" value="'.htmlspecialchars($_SESSION['csrf_token']).'">
                        <input type="hidden" name="action" value="soft_delete">
                        <input type="hidden" name="id" value="'.$n['id'].'">
                        <button class="btn btn-sm btn-outline-danger" title="أرشفة"><i class="fa-solid fa-box-archive"></i></button>
                      </form>';
              } else {
                echo '<form method="post" class="d-inline">
                        <input type="hidden" name="csrf" value="'.htmlspecialchars($_SESSION['csrf_token']).'">
                        <input type="hidden" name="action" value="restore">
                        <input type="hidden" name="id" value="'.$n['id'].'">
                        <button class="btn btn-sm btn-outline-success" title="استعادة"><i class="fa-solid fa-rotate-left"></i></button>
                      </form>';
              }
            }
            echo '</div></div>';

            // الأطفال (قائمة قابلة للسحب أيضًا، وتحمل parent_id)
            if (!empty($n['children'])) {
              echo '<ul class="ps-0 sortable" data-parent="'.$n['id'].'">';
              foreach ($n['children'] as $ch) {
                renderTree([$ch], $level+1, $hasProducts);
              }
              echo '</ul>';
            }
            echo '</li>';
          }
          echo '</ul>';
        }
        renderTree($tree, 0, $hasProducts);
        ?>
      </div>

    </div>
  </div>
</div>

<!-- Modal: Create/Edit Category -->
<div class="modal fade" id="catModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-layer-group me-2"></i> <span id="modalTitle">تصنيف جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <form method="post" enctype="multipart/form-data" class="modal-body row g-3">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save">
        <input type="hidden" name="id" id="f_id">

        <div class="col-md-6">
          <label class="form-label">الاسم (عربي) *</label>
          <input type="text" class="form-control" name="name_ar" id="f_name_ar" required>
        </div>
        <div class="col-md-6">
          <label class="form-label">الاسم (إنجليزي)</label>
          <input type="text" class="form-control" name="name_en" id="f_name_en">
        </div>

        <div class="col-md-6">
          <label class="form-label">التصنيف الأب</label>
          <select class="form-select" name="parent_id" id="f_parent">
            <option value="">— رئيسي —</option>
            <?= optionTreeHtml($fullTree) ?>
          </select>
        </div>

        <div class="col-md-6">
          <label class="form-label">Slug (اختياري)</label>
          <input type="text" class="form-control" name="slug" id="f_slug" placeholder="لو تركته فارغًا سيُولَّد تلقائيًا">
        </div>

        <div class="col-12">
          <label class="form-label">الوصف (اختياري)</label>
          <textarea class="form-control" name="description" id="f_desc" rows="2"></textarea>
        </div>

        <div class="col-md-6">
          <label class="form-label">أيقونة/صورة (حد 3MB)</label>
          <input type="file" class="form-control" name="image" accept=".jpg,.jpeg,.png,.webp,image/*">
        </div>
        <div class="col-md-3">
          <label class="form-label">الترتيب</label>
          <input type="number" class="form-control" name="sort_order" id="f_sort" value="0">
        </div>
        <div class="col-md-3 d-flex align-items-center">
          <div class="form-check mt-4">
            <input type="checkbox" class="form-check-input" id="f_active" name="is_active" checked>
            <label class="form-check-label" for="f_active">مفعل</label>
          </div>
        </div>

        <div class="col-12 text-center mt-2">
          <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          <button type="button" class="btn btn-outline-secondary ms-2" data-bs-dismiss="modal">إلغاء</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<!-- SortableJS للسحب والإفلات -->
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js"></script>
<script>
  // Dark mode
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode") === "true") { body.classList.add("dark"); darkSwitch.textContent = "☀️"; }
  darkSwitch?.addEventListener("click", ()=> {
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  const catModal = new bootstrap.Modal(document.getElementById('catModal'));
  const modalTitle = document.getElementById('modalTitle');

  function openCreate(){
    modalTitle.textContent = 'تصنيف جديد';
    document.getElementById('f_id').value      = '';
    document.getElementById('f_name_ar').value = '';
    document.getElementById('f_name_en').value = '';
    document.getElementById('f_parent').value  = '';
    document.getElementById('f_slug').value    = '';
    document.getElementById('f_desc').value    = '';
    document.getElementById('f_sort').value    = '0';
    document.getElementById('f_active').checked= true;
    catModal.show();
  }
  function openEdit(node){
    modalTitle.textContent = 'تعديل تصنيف';
    document.getElementById('f_id').value      = node.id ?? '';
    document.getElementById('f_name_ar').value = node.name_ar ?? '';
    document.getElementById('f_name_en').value = node.name_en ?? '';
    document.getElementById('f_parent').value  = node.parent_id ?? '';
    document.getElementById('f_slug').value    = node.slug ?? '';
    document.getElementById('f_desc').value    = node.description ?? '';
    document.getElementById('f_sort').value    = node.sort_order ?? 0;
    document.getElementById('f_active').checked= (parseInt(node.is_active ?? 1) === 1);
    catModal.show();
  }

  // سحب/إفلات لإعادة الترتيب داخل نفس المستوى
  const lists = document.querySelectorAll('.sortable');
  lists.forEach(list => {
    new Sortable(list, {
      animation: 150,
      ghostClass: 'sortable-placeholder',
      handle: '.cat-item', // كل العنصر
      onEnd: function(evt){
        // نجمع الترتيب الجديد لنفس القائمة (نفس الأب)
        const parent = list.getAttribute('data-parent') || 'root';
        const ids = Array.from(list.querySelectorAll(':scope > .cat-item')).map(li => li.getAttribute('data-id'));
        // إرسال عبر POST (Fetch) لحفظ الترتيب
        const fd = new FormData();
        fd.append('csrf', '<?= $csrf ?>');
        fd.append('action', 'reorder_seq');
        fd.append('parent_id', parent);
        ids.forEach(id => fd.append('order[]', id));
        fetch(location.href, { method: 'POST', body: fd })
          .then(r => r.text())
          .then(() => {
            // ممكن تعرض توست بسيط إن حبيت
            console.log('Reorder saved');
          })
          .catch(console.error);
      }
    });
  });
</script>
</body>
</html>
