<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ========= إعدادات / صلاحيات بسيطة ========= */
$canEdit = true;

/* ========= CSRF ========= */
if (empty($_SESSION['csrf_token'])) {
  $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
}
$csrf = $_SESSION['csrf_token'];

/* ========= Helpers ========= */
function clean_str(?string $s, int $max=255): ?string {
  $s = trim((string)$s);
  if ($s==='') return null;
  return mb_substr($s, 0, $max);
}
function combo_hash(array $pairs): string {
  // $pairs = [attribute_id => attribute_value_id, ...]
  ksort($pairs, SORT_NUMERIC);
  $buf = [];
  foreach ($pairs as $aid=>$vid) { $buf[] = $aid.':'.$vid; }
  return sha1(implode('|', $buf));
}
function hasTable(PDO $pdo, string $t): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?");
  $st->execute([$t]); return (int)$st->fetchColumn() > 0;
}
function hasColumn(PDO $pdo, string $t, string $c): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name = ? AND column_name = ?");
  $st->execute([$t,$c]); return (int)$st->fetchColumn() > 0;
}

/* ========= اكتشاف جدول المنتجات (products أو items) ========= */
$PRODUCTS = [
  'table' => null,
  'id_col' => null,
  'name_col' => null
];
if (hasTable($pdo, 'products')) {
  $PRODUCTS['table'] = 'products';
  $PRODUCTS['id_col'] = hasColumn($pdo,'products','id') ? 'id' : 'ProductID';
  $PRODUCTS['name_col'] = hasColumn($pdo,'products','name_ar') ? 'name_ar' :
                          (hasColumn($pdo,'products','name') ? 'name' :
                          (hasColumn($pdo,'products','ProductName') ? 'ProductName' : 'name_ar'));
} elseif (hasTable($pdo, 'items')) {
  $PRODUCTS['table'] = 'items';
  $PRODUCTS['id_col'] = hasColumn($pdo,'items','id') ? 'id' : (hasColumn($pdo,'items','ItemID') ? 'ItemID' : 'id');
  $PRODUCTS['name_col'] = hasColumn($pdo,'items','ItemName') ? 'ItemName' :
                          (hasColumn($pdo,'items','name_ar') ? 'name_ar' :
                          (hasColumn($pdo,'items','name') ? 'name' : 'ItemName'));
} else {
  // كحل مؤقت: أنشئ قائمة فارغة وننبه المستخدم
  $PRODUCTS['table'] = null;
}

/* ========= جلب المنتجات ========= */
$products = [];
if ($PRODUCTS['table']) {
  $sqlP = "SELECT {$PRODUCTS['id_col']} AS id, {$PRODUCTS['name_col']} AS name
           FROM {$PRODUCTS['table']} ORDER BY name LIMIT 1000";
  $products = $pdo->query($sqlP)->fetchAll(PDO::FETCH_KEY_PAIR); // id => name
}

/* ========= جلب السمات وقيمها ========= */
$attributes = $pdo->query("SELECT * FROM attributes WHERE is_active=1 ORDER BY sort_order, id")->fetchAll(PDO::FETCH_ASSOC);
$attrValues = [];
if ($attributes) {
  $st = $pdo->query("SELECT * FROM attribute_values WHERE is_active=1 ORDER BY attribute_id, sort_order, id");
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $v) {
    $attrValues[(int)$v['attribute_id']][] = $v;
  }
}

/* ========= جلب الفروع ========= */
$branches = [];
if (hasTable($pdo, 'Branches')) {
  $branches = $pdo->query("SELECT BranchID, BranchName FROM Branches ORDER BY BranchName")->fetchAll(PDO::FETCH_KEY_PAIR);
}

/* ========= Actions ========= */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).'; 
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية التعديل.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action === 'generate') {
        // توليد تركيبات لمنتج معيّن
        $product_id  = (int)($_POST['product_id'] ?? 0);
        if ($product_id <= 0) throw new RuntimeException('اختر منتجًا أولًا.');
        $sku_prefix  = clean_str($_POST['sku_prefix'] ?? '', 40) ?: 'SKU';
        $price       = (float)($_POST['price'] ?? 0);
        $compare     = ($_POST['compare_price'] ?? '') !== '' ? (float)$_POST['compare_price'] : null;
        $cost        = ($_POST['cost'] ?? '') !== '' ? (float)$_POST['cost'] : null;
        $weight      = ($_POST['weight'] ?? '') !== '' ? (float)$_POST['weight'] : null;
        $is_active   = isset($_POST['is_active']) ? 1 : 0;

        // attr_values[attribute_id][] = [value_ids...]
        $posted = $_POST['attr_values'] ?? [];
        $sets = []; // قائمة قوائم القيم
        $codesMap = []; // attribute_id => [value_id => code]
        foreach ($attributes as $a) {
          $aid = (int)$a['id'];
          if (!empty($posted[$aid]) && is_array($posted[$aid])) {
            $vals = array_values(array_filter(array_map('intval', $posted[$aid])));
            if ($vals) {
              $sets[] = $vals;
              // خريطة الأكواد
              foreach ($attrValues[$aid] ?? [] as $v) {
                $codesMap[$aid][$v['id']] = $v['code'] ?: $v['value_en'] ?: $v['value_ar'];
              }
            }
          }
        }
        if (!$sets) throw new RuntimeException('اختر قيمًا واحدة على الأقل لأي سمة.');

        // Cartesian product
        $combinations = [[]];
        foreach ($sets as $arr) {
          $tmp = [];
          foreach ($combinations as $base) {
            foreach ($arr as $x) { $tmp[] = array_merge($base, [$x]); }
          }
          $combinations = $tmp;
        }

        $inserted=0; $skipped=0;
        foreach ($combinations as $combo) {
          // كوّن pairs attribute_id => attribute_value_id بنفس ترتيب attributes المُختارة
          $pairs = [];
          $sku_parts = [$sku_prefix];
          $i = 0;
          foreach ($attributes as $a) {
            $aid = (int)$a['id'];
            if (!empty($posted[$aid])) {
              $vid = (int)$combo[$i++] ?? null;
              if (!$vid) continue;
              $pairs[$aid] = $vid;
              $code = $codesMap[$aid][$vid] ?? $vid;
              $sku_parts[] = strtoupper(preg_replace('~\W+~','',$code));
            }
          }
          if (!$pairs) continue;
          $hash = combo_hash($pairs);

          // تحقق عدم وجودها
          $st = $pdo->prepare("SELECT id FROM product_variants WHERE product_id=? AND combo_hash=? LIMIT 1");
          $st->execute([$product_id,$hash]);
          $exists = $st->fetchColumn();
          if ($exists) { $skipped++; continue; }

          $sku = implode('-', $sku_parts);
          // تأكد من تفرّد الـ SKU
          $checkSku = $pdo->prepare("SELECT 1 FROM product_variants WHERE sku=? LIMIT 1");
          $trySku = $sku; $n=2;
          $checkSku->execute([$trySku]);
          while ($checkSku->fetch()) { $trySku = $sku.'-'.$n++; $checkSku->execute([$trySku]); }
          $sku = $trySku;

          // أدخل المتغيّر
          $ins = $pdo->prepare("INSERT INTO product_variants (product_id, sku, barcode, price, compare_price, cost, weight, is_active, combo_hash)
                                VALUES (?,?,?,?,?,?,?,?,?)");
          $ins->execute([$product_id, $sku, null, $price, $compare, $cost, $weight, $is_active, $hash]);
          $vid = (int)$pdo->lastInsertId();

          // أدخل قيم السمات
          $insV = $pdo->prepare("INSERT INTO product_variant_values (variant_id, attribute_id, attribute_value_id)
                                 VALUES (?,?,?)");
          foreach ($pairs as $aid=>$valId) { $insV->execute([$vid, $aid, $valId]); }

          // مخزون ابتدائي لكل الفروع (اختياري)
          if (!empty($_POST['init_stock']) && is_array($_POST['init_stock'])) {
            $insS = $pdo->prepare("INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
                                   VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE quantity=VALUES(quantity), min_quantity=VALUES(min_quantity)");
            foreach ($_POST['init_stock'] as $bid=>$qty) {
              $qty = (int)$qty;
              $min = (int)($_POST['init_min'][$bid] ?? 0);
              $insS->execute([$vid, (int)$bid, $qty, $min]);
            }
          }

          $inserted++;
        }
        $success = "✔️ تم إنشاء {$inserted} متغيّر" . ($skipped ? " — تم تخطي {$skipped} موجودة بالفعل" : "");

      } elseif ($action === 'save_variant') {
        $id = (int)($_POST['variant_id'] ?? 0);
        if ($id<=0) throw new RuntimeException('متغيّر غير معروف.');
        $sku     = clean_str($_POST['sku'] ?? '', 80);
        $barcode = clean_str($_POST['barcode'] ?? '', 64);
        $price   = (float)($_POST['price'] ?? 0);
        $compare = ($_POST['compare_price'] ?? '')!=='' ? (float)$_POST['compare_price'] : null;
        $cost    = ($_POST['cost'] ?? '')!=='' ? (float)$_POST['cost'] : null;
        $weight  = ($_POST['weight'] ?? '')!=='' ? (float)$_POST['weight'] : null;
        $active  = isset($_POST['is_active']) ? 1 : 0;

        // تحديث الأساسيات
        $st = $pdo->prepare("UPDATE product_variants SET sku=?, barcode=?, price=?, compare_price=?, cost=?, weight=?, is_active=? WHERE id=? AND deleted_at IS NULL");
        $st->execute([$sku,$barcode,$price,$compare,$cost,$weight,$active,$id]);

        // تحديث مخزون الفروع
        if (isset($_POST['stock']) && is_array($_POST['stock'])) {
          $up = $pdo->prepare("INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
                               VALUES (?,?,?,?) ON DUPLICATE KEY UPDATE quantity=VALUES(quantity), min_quantity=VALUES(min_quantity)");
          foreach ($_POST['stock'] as $bid=>$qty) {
            $qty = (int)$qty;
            $min = (int)($_POST['min'][$bid] ?? 0);
            $up->execute([$id,(int)$bid,$qty,$min]);
          }
        }

        // رفع صور (اختياري)
        if (!empty($_FILES['images']) && is_array($_FILES['images']['name'])) {
          $uploadDir = __DIR__ . '/uploads/variants/';
          if (!is_dir($uploadDir)) @mkdir($uploadDir,0777,true);
          $allowed = ['jpg','jpeg','png','webp'];
          $c = $pdo->prepare("SELECT COALESCE(MAX(sort_order),0)+1 FROM product_variant_images WHERE variant_id=?");
          $c->execute([$id]); $sort = (int)$c->fetchColumn();
          for ($i=0;$i<count($_FILES['images']['name']);$i++){
            if (($_FILES['images']['error'][$i] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) continue;
            $ext = strtolower(pathinfo($_FILES['images']['name'][$i], PATHINFO_EXTENSION));
            if (!in_array($ext,$allowed,true)) continue;
            if ((int)$_FILES['images']['size'][$i] > 3*1024*1024) continue;
            $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
            if (move_uploaded_file($_FILES['images']['tmp_name'][$i], $uploadDir.$name)) {
              $pdo->prepare("INSERT INTO product_variant_images (variant_id,image,sort_order) VALUES (?,?,?)")
                  ->execute([$id,$name,$sort++]);
            }
          }
        }

        $success = '✔️ تم حفظ المتغيّر';

      } elseif ($action === 'toggle' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE product_variants SET is_active = 1 - is_active WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم تغيير الحالة';

      } elseif ($action === 'soft_delete' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE product_variants SET is_active=0, deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة المتغيّر';

      } elseif ($action === 'restore' && isset($_POST['id'])) {
        $id=(int)$_POST['id'];
        $pdo->prepare("UPDATE product_variants SET deleted_at=NULL, is_active=1 WHERE id=?")->execute([$id]);
        $success = 'تم الاستعادة';

      } elseif ($action === 'reorder_images') {
        $vid = (int)($_POST['variant_id'] ?? 0);
        $order = $_POST['order'] ?? [];
        if ($vid>0 && is_array($order)) {
          $i=0;
          $st = $pdo->prepare("UPDATE product_variant_images SET sort_order=? WHERE id=? AND variant_id=?");
          foreach ($order as $imgId) { $st->execute([$i++,(int)$imgId,$vid]); }
          $success = 'تم حفظ ترتيب الصور';
        }
      } elseif ($action === 'delete_image') {
        $imgId = (int)($_POST['image_id'] ?? 0);
        $row = $pdo->prepare("SELECT variant_id,image FROM product_variant_images WHERE id=?");
        $row->execute([$imgId]); $img = $row->fetch(PDO::FETCH_ASSOC);
        if ($img) {
          $pdo->prepare("DELETE FROM product_variant_images WHERE id=?")->execute([$imgId]);
          $path = __DIR__ . '/uploads/variants/' . $img['image'];
          if (is_file($path)) @unlink($path);
          $success = 'تم حذف الصورة';
        }
      }
    } catch (Throwable $e) {
      $error = 'خطأ: '.htmlspecialchars($e->getMessage(), ENT_QUOTES, 'UTF-8');
    }
  }
}

/* ========= فلاتر/بحث ========= */
$q = trim($_GET['q'] ?? '');
$filterProduct = (int)($_GET['product_id'] ?? 0);
$showInactive = isset($_GET['show_inactive']) ? 1 : 0;
$showArchived = isset($_GET['show_archived']) ? 1 : 0;

/* ========= جلب قائمة المتغيّرات ========= */
$where = [];
$params = [];
$where[] = $showArchived ? "1=1" : "v.deleted_at IS NULL";
if (!$showInactive) $where[] = "v.is_active=1";
if ($filterProduct > 0) { $where[] = "v.product_id=:pid"; $params[':pid']=$filterProduct; }
if ($q !== '') {
  $where[] = "(v.sku LIKE :q OR v.barcode LIKE :q)";
  $params[':q'] = "%$q%";
}
$whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

/* اسم عمود المنتج للعرض */
$productNameCol = $PRODUCTS['name_col'] ?? 'name';
$productIdCol   = $PRODUCTS['id_col'] ?? 'id';
$productTable   = $PRODUCTS['table'] ?: 'products';

$variants = [];
try {
  $sql = "SELECT v.*,
                 p.`$productNameCol` AS ProductName,
                 COALESCE(SUM(s.quantity),0) AS total_qty
          FROM product_variants v
          LEFT JOIN `$productTable` p ON p.`$productIdCol` = v.product_id
          LEFT JOIN product_variant_stock s ON s.variant_id = v.id
          $whereSql
          GROUP BY v.id
          ORDER BY v.id DESC
          LIMIT 1000";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $variants = $st->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $e) {
  $variants = [];
}

/* قيم السمات لكل متغيّر للعرض */
$attrMap = [];
if ($variants) {
  $ids = array_column($variants,'id');
  $in = implode(',', array_fill(0, count($ids), '?'));
  $st = $pdo->prepare("SELECT pv.variant_id, a.name_ar AS attr, av.value_ar AS val, a.id AS aid, av.id AS vid
                       FROM product_variant_values pv
                       JOIN attributes a ON a.id=pv.attribute_id
                       JOIN attribute_values av ON av.id=pv.attribute_value_id
                       WHERE pv.variant_id IN ($in)
                       ORDER BY a.sort_order, av.sort_order");
  $st->execute($ids);
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $attrMap[(int)$r['variant_id']][] = $r;
  }
}

/* مخزون الفروع لكل متغيّر (للمودال) */
$stocksMap = [];
if ($variants && $branches) {
  $ids = array_column($variants,'id');
  $in = implode(',', array_fill(0, count($ids), '?'));
  $st = $pdo->prepare("SELECT variant_id, branch_id, quantity, min_quantity FROM product_variant_stock WHERE variant_id IN ($in)");
  $st->execute($ids);
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $stocksMap[(int)$r['variant_id']][(int)$r['branch_id']] = ['q'=>(int)$r['quantity'],'m'=>(int)$r['min_quantity']];
  }
}

/* صور كل متغيّر (للمودال) */
$imgsMap = [];
if ($variants) {
  $ids = array_column($variants,'id');
  $in = implode(',', array_fill(0, count($ids), '?'));
  $st = $pdo->prepare("SELECT id, variant_id, image, sort_order FROM product_variant_images WHERE variant_id IN ($in) ORDER BY sort_order, id");
  $st->execute($ids);
  foreach ($st->fetchAll(PDO::FETCH_ASSOC) as $r) {
    $imgsMap[(int)$r['variant_id']][] = $r;
  }
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>المتغيّرات (Variants)</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#ffffff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{
      background: linear-gradient(90deg, var(--p1), var(--p2), var(--p3), var(--p4));
      padding:14px 18px; color:#1f2a37; display:flex; align-items:center; gap:12px; justify-content:space-between;
      box-shadow:0 6px 24px rgba(0,0,0,.15);
    }
    .user-info{display:flex; align-items:center; gap:10px; background:rgba(255,255,255,.6); padding:6px 10px; border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}

    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .img-thumb{width:48px;height:48px;object-fit:cover;border-radius:8px;border:1px solid var(--border)}
    .sortable-placeholder{border:2px dashed var(--primary); border-radius:12px; height:56px; margin:8px 0}
    .muted{color:var(--muted)}
    .table>:not(caption)>*>*{background:transparent}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= htmlspecialchars($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-palette me-2"></i> المتغيّرات (Variants)</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- تنبيه لو مفيش جدول منتجات -->
  <?php if (!$PRODUCTS['table']): ?>
    <div class="alert alert-warning">⚠️ لم يتم العثور على جدول منتجات (products/items). رجاءً أنشئ واحدًا أو أخبرني باسم الجدول وأعمدته لأعدّل الصفحة.</div>
  <?php endif; ?>

  <!-- مولّد التركيبات -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <div class="fw-bold"><i class="fa-solid fa-wand-magic-sparkles me-2"></i> إنشاء متغيّرات جماعيًا</div>
    </div>
    <div class="p-3">
      <form method="post">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="generate">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">المنتج *</label>
            <select class="form-select" name="product_id" required>
              <option value="">اختر المنتج</option>
              <?php foreach ($products as $pid=>$pname): ?>
                <option value="<?= (int)$pid ?>"><?= htmlspecialchars($pname, ENT_QUOTES, 'UTF-8') ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-2">
            <label class="form-label">SKU Prefix</label>
            <input type="text" class="form-control" name="sku_prefix" placeholder="مثال: TSHIRT">
          </div>
          <div class="col-md-2">
            <label class="form-label">السعر</label>
            <input type="number" step="0.01" class="form-control" name="price" value="0">
          </div>
          <div class="col-md-2">
            <label class="form-label">سعر مقارنة</label>
            <input type="number" step="0.01" class="form-control" name="compare_price">
          </div>
          <div class="col-md-2">
            <label class="form-label">التكلفة</label>
            <input type="number" step="0.01" class="form-control" name="cost">
          </div>
          <div class="col-md-2">
            <label class="form-label">الوزن</label>
            <input type="number" step="0.001" class="form-control" name="weight">
          </div>
          <div class="col-md-2 d-flex align-items-end">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" name="is_active" id="gen_active" checked>
              <label class="form-check-label" for="gen_active">مفعل</label>
            </div>
          </div>

          <!-- قيم السمات -->
          <?php foreach ($attributes as $a): $aid=(int)$a['id']; ?>
            <div class="col-12">
              <label class="form-label"><?= htmlspecialchars($a['name_ar']) ?>:</label>
              <div class="d-flex flex-wrap gap-2">
                <?php foreach (($attrValues[$aid] ?? []) as $v): ?>
                  <label class="form-check form-check-inline">
                    <input class="form-check-input" type="checkbox" name="attr_values[<?= $aid ?>][]" value="<?= (int)$v['id'] ?>">
                    <span class="form-check-label">
                      <?php if ($a['code']==='color' && $v['color_hex']): ?>
                        <span class="me-1" style="display:inline-block;width:14px;height:14px;border:1px solid #ccc;background:<?= htmlspecialchars($v['color_hex']) ?>;"></span>
                      <?php endif; ?>
                      <?= htmlspecialchars($v['value_ar']) ?>
                      <?php if ($v['code']) echo ' ('.htmlspecialchars($v['code']).')'; ?>
                    </span>
                  </label>
                <?php endforeach; ?>
              </div>
            </div>
          <?php endforeach; ?>

          <!-- مخزون ابتدائي اختياري -->
          <?php if ($branches): ?>
          <div class="col-12">
            <div class="alert alert-secondary py-2"><strong>مخزون ابتدائي (اختياري)</strong></div>
            <div class="row g-2">
              <?php foreach ($branches as $bid=>$bname): ?>
                <div class="col-md-3">
                  <label class="form-label"><?= htmlspecialchars($bname) ?> — الكمية</label>
                  <input type="number" class="form-control" name="init_stock[<?= (int)$bid ?>]" value="0">
                </div>
                <div class="col-md-3">
                  <label class="form-label">الحد الأدنى</label>
                  <input type="number" class="form-control" name="init_min[<?= (int)$bid ?>]" value="0">
                </div>
              <?php endforeach; ?>
            </div>
          </div>
          <?php endif; ?>

          <div class="col-12 text-end">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-wand-magic-sparkles"></i> توليد المتغيّرات</button>
          </div>
        </div>
      </form>
    </div>
  </div>

  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث بـ SKU/Barcode" value="<?= htmlspecialchars($q) ?>">
        <select class="form-select" name="product_id" style="min-width:240px">
          <option value="">كل المنتجات</option>
          <?php foreach ($products as $pid=>$pname): ?>
            <option value="<?= (int)$pid ?>" <?= $filterProduct===$pid ? 'selected':''; ?>><?= htmlspecialchars($pname) ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_inactive" name="show_inactive" value="1" <?= $showInactive?'checked':''; ?>>
          <label class="form-check-label" for="show_inactive">إظهار غير المفعل</label>
        </div>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $filterProduct || $showInactive || $showArchived): ?>
        <a href="variants.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>
      <div class="muted">إجمالي: <span class="pill"><?= count($variants) ?></span></div>
    </div>
  </div>

  <!-- جدول المتغيّرات -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>المنتج</th>
              <th>السمات</th>
              <th>SKU</th>
              <th>Barcode</th>
              <th>السعر</th>
              <th>الكمية الكلية</th>
              <th>الحالة</th>
              <th>صور</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$variants): ?>
            <tr><td colspan="10" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($variants as $v): 
            $vid = (int)$v['id'];
            $attrs = $attrMap[$vid] ?? [];
            $attrsStr = $attrs ? implode(' | ', array_map(fn($r)=>$r['attr'].': '.$r['val'], $attrs)) : '—';
            $imgs = $imgsMap[$vid] ?? [];
          ?>
            <tr data-vid="<?= $vid ?>"
                data-variant='<?= htmlspecialchars(json_encode($v), ENT_QUOTES, "UTF-8") ?>'
                data-stocks='<?= htmlspecialchars(json_encode($stocksMap[$vid] ?? []), ENT_QUOTES, "UTF-8") ?>'
                data-images='<?= htmlspecialchars(json_encode($imgs), ENT_QUOTES, "UTF-8") ?>'>
              <td><?= $vid ?></td>
              <td><?= htmlspecialchars($v['ProductName'] ?? '—', ENT_QUOTES, 'UTF-8') ?></td>
              <td class="text-start"><?= htmlspecialchars($attrsStr, ENT_QUOTES, 'UTF-8') ?></td>
              <td><span class="pill"><?= htmlspecialchars($v['sku'], ENT_QUOTES, 'UTF-8') ?></span></td>
              <td><?= htmlspecialchars($v['barcode'] ?? '—', ENT_QUOTES, 'UTF-8') ?></td>
              <td><?= number_format((float)$v['price'], 2) ?></td>
              <td><?= (int)($v['total_qty'] ?? 0) ?></td>
              <td><?= ((int)$v['is_active']===1) ? '<span class="badge bg-success">مفعل</span>' : '<span class="badge bg-warning text-dark">غير مفعل</span>' ?></td>
              <td>
                <?php if ($imgs): ?>
                  <?php foreach (array_slice($imgs,0,2) as $im): ?>
                    <img class="img-thumb me-1" src="uploads/variants/<?= htmlspecialchars($im['image']) ?>" alt="">
                  <?php endforeach; ?>
                  <?php if (count($imgs)>2): ?><span class="pill">+<?= count($imgs)-2 ?></span><?php endif; ?>
                <?php else: ?>—<?php endif; ?>
              </td>
              <td>
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(this)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <form method="post" class="d-inline">
                    <input type="hidden" name="csrf" value="<?= $csrf ?>">
                    <input type="hidden" name="action" value="toggle">
                    <input type="hidden" name="id" value="<?= $vid ?>">
                    <button class="btn btn-sm btn-outline-warning"><i class="fa-solid fa-toggle-on"></i></button>
                  </form>
                  <?php if ($v['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة المتغيّر؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= $vid ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= $vid ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Edit Variant -->
<div class="modal fade" id="variantModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-pen-to-square me-2"></i> تعديل متغيّر</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
      </div>
      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_variant">
        <input type="hidden" name="variant_id" id="f_vid">
        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">SKU</label>
            <input type="text" class="form-control" name="sku" id="f_sku">
          </div>
          <div class="col-md-4">
            <label class="form-label">Barcode</label>
            <input type="text" class="form-control" name="barcode" id="f_barcode">
          </div>
          <div class="col-md-2">
            <label class="form-label">السعر</label>
            <input type="number" step="0.01" class="form-control" name="price" id="f_price">
          </div>
          <div class="col-md-2">
            <label class="form-label">سعر مقارنة</label>
            <input type="number" step="0.01" class="form-control" name="compare_price" id="f_compare">
          </div>
          <div class="col-md-2">
            <label class="form-label">التكلفة</label>
            <input type="number" step="0.01" class="form-control" name="cost" id="f_cost">
          </div>
          <div class="col-md-2">
            <label class="form-label">الوزن</label>
            <input type="number" step="0.001" class="form-control" name="weight" id="f_weight">
          </div>
          <div class="col-md-2 d-flex align-items-end">
            <div class="form-check">
              <input class="form-check-input" type="checkbox" id="f_active" name="is_active">
              <label class="form-check-label" for="f_active">مفعل</label>
            </div>
          </div>

          <?php if ($branches): ?>
          <div class="col-12">
            <div class="alert alert-secondary py-2"><strong>المخزون لكل فرع</strong></div>
            <div class="row g-2">
              <?php foreach ($branches as $bid=>$bname): ?>
                <div class="col-md-3">
                  <label class="form-label"><?= htmlspecialchars($bname) ?> — كمية</label>
                  <input type="number" class="form-control" name="stock[<?= (int)$bid ?>]" id="stk_<?= (int)$bid ?>" value="0">
                </div>
                <div class="col-md-3">
                  <label class="form-label">حد أدنى</label>
                  <input type="number" class="form-control" name="min[<?= (int)$bid ?>]" id="min_<?= (int)$bid ?>" value="0">
                </div>
              <?php endforeach; ?>
            </div>
          </div>
          <?php endif; ?>

          <div class="col-12">
            <label class="form-label">صور (متعددة، حتى 3MB للصورة)</label>
            <input type="file" class="form-control" name="images[]" multiple accept=".jpg,.jpeg,.png,.webp,image/*">
          </div>

          <div class="col-12">
            <div class="d-flex flex-wrap gap-2" id="img_list">
              <!-- صور المتغيّر تُملأ بالـ JS -->
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>

      <!-- نموذج ترتيب/حذف الصور (AJAX بسيط) -->
      <form id="img_actions" method="post" class="d-none">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="">
        <input type="hidden" name="variant_id" value="">
        <input type="hidden" name="order[]" value="">
        <input type="hidden" name="image_id" value="">
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.2/Sortable.min.js"></script>
<script>
  // Dark mode
  const body = document.body;
  const darkSwitch = document.getElementById("darkSwitch");
  if (localStorage.getItem("darkMode") === "true") { body.classList.add("dark"); darkSwitch.textContent = "☀️"; }
  darkSwitch?.addEventListener("click", ()=> {
    body.classList.toggle("dark");
    const isDark = body.classList.contains("dark");
    localStorage.setItem("darkMode", isDark);
    darkSwitch.textContent = isDark ? "☀️" : "🌙";
  });

  const variantModal = new bootstrap.Modal(document.getElementById('variantModal'));

  function openEdit(btn){
    const tr = btn.closest('tr');
    const v  = JSON.parse(tr.getAttribute('data-variant'));
    const stocks = JSON.parse(tr.getAttribute('data-stocks') || '{}');
    const images = JSON.parse(tr.getAttribute('data-images') || '[]');

    document.getElementById('f_vid').value    = v.id;
    document.getElementById('f_sku').value    = v.sku ?? '';
    document.getElementById('f_barcode').value= v.barcode ?? '';
    document.getElementById('f_price').value  = v.price ?? 0;
    document.getElementById('f_compare').value= v.compare_price ?? '';
    document.getElementById('f_cost').value   = v.cost ?? '';
    document.getElementById('f_weight').value = v.weight ?? '';
    document.getElementById('f_active').checked = (parseInt(v.is_active ?? 1) === 1);

    // مخزون الفروع
    <?php foreach ($branches as $bid=>$bname): ?>
      (document.getElementById('stk_<?= (int)$bid ?>')||{}).value = (stocks['<?= (int)$bid ?>']?.q ?? 0);
      (document.getElementById('min_<?= (int)$bid ?>')||{}).value = (stocks['<?= (int)$bid ?>']?.m ?? 0);
    <?php endforeach; ?>

    // صور
    const imgList = document.getElementById('img_list');
    imgList.innerHTML = '';
    images.forEach(im => {
      const wrap = document.createElement('div');
      wrap.className = 'd-flex flex-column align-items-center';
      wrap.setAttribute('data-imgid', im.id);
      wrap.innerHTML = `
        <img class="img-thumb" src="uploads/variants/${im.image}" alt="">
        <button type="button" class="btn btn-sm btn-outline-danger mt-1" onclick="deleteImage(${im.id})"><i class="fa-solid fa-trash"></i></button>
      `;
      imgList.appendChild(wrap);
    });

    // Sortable images
    new Sortable(imgList, {
      animation: 150,
      onEnd: function(){
        // أرسل ترتيب الصور
        const ids = Array.from(imgList.children).map(el => el.getAttribute('data-imgid'));
        const fd = new FormData();
        fd.append('csrf','<?= $csrf ?>');
        fd.append('action','reorder_images');
        fd.append('variant_id', v.id);
        ids.forEach(id => fd.append('order[]', id));
        fetch(location.href,{method:'POST', body:fd});
      }
    });

    variantModal.show();
  }

  function deleteImage(imageId){
    if (!confirm('حذف الصورة؟')) return;
    const fd = new FormData();
    fd.append('csrf','<?= $csrf ?>');
    fd.append('action','delete_image');
    fd.append('image_id', imageId);
    fetch(location.href,{method:'POST', body:fd})
      .then(()=> location.reload());
  }
</script>
</body>
</html>
