<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ===== Helpers عامة ===== */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function hasColumn(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
  $st->execute([$table,$col]); return (int)$st->fetchColumn() > 0;
}
function money($n){ return number_format((float)$n,2); }
function clean_str(?string $s, int $max=255): ?string { $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }

/* صلاحيات */
$canEdit = true;

/* CSRF */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ===== كشف أعمدة جدول الفروع ===== */
$BR_TBL = 'branches';
function branchCols(PDO $pdo): array {
  $candsId = ['id','branch_id','BranchID','ID'];
  $candsNm = ['name','branch_name','BranchName','Name'];
  $idCol = 'id'; $nmCol = 'name';
  foreach($candsId as $c){ if(hasColumn($pdo,'branches',$c)) { $idCol=$c; break; } }
  foreach($candsNm as $c){ if(hasColumn($pdo,'branches',$c)) { $nmCol=$c; break; } }
  return [$idCol, $nmCol];
}
[$BR_ID_COL, $BR_NAME_COL] = branchCols($pdo);

/* ===== تأكيد جداول التحويلات (إن لم توجد) ===== */
$pdo->exec("
CREATE TABLE IF NOT EXISTS stock_transfers (
  id INT AUTO_INCREMENT PRIMARY KEY,
  from_branch_id INT NOT NULL,
  to_branch_id   INT NOT NULL,
  status ENUM('draft','requested','partial','transferred','cancelled') NOT NULL DEFAULT 'draft',
  transfer_date DATE NOT NULL,
  notes TEXT NULL,
  attachment VARCHAR(255) NULL,
  deleted_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS stock_transfer_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  transfer_id INT NOT NULL,
  variant_id INT NOT NULL,
  description VARCHAR(255) NULL,
  qty INT NOT NULL DEFAULT 0,
  processed_qty INT NOT NULL DEFAULT 0,
  line_note VARCHAR(255) NULL,
  CONSTRAINT fk_sti_header FOREIGN KEY (transfer_id) REFERENCES stock_transfers(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

/* ===== فروع ===== */
try {
  $branches = $pdo->query("SELECT `$BR_ID_COL` AS BranchID, `$BR_NAME_COL` AS BranchName FROM `$BR_TBL` ORDER BY `$BR_NAME_COL`")
                  ->fetchAll(PDO::FETCH_KEY_PAIR);
} catch(Throwable $e){ $branches = []; }

/* ===== متغيّرات للبحث السريع ===== */
$variants = $pdo->query("
  SELECT v.id, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-'),' | ',v.sku) AS label
  FROM product_variants v
  JOIN products p ON p.id=v.product_id
  WHERE v.deleted_at IS NULL
  ORDER BY p.name_ar, v.id
  LIMIT 2000
")->fetchAll(PDO::FETCH_ASSOC);

function last_purchase_cost(PDO $pdo, int $variantId): float {
  try {
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]);
    $x=$st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  } catch(Throwable $e){ return 0.0; }
}

/* ===== JSON endpoint: تحميل تحويل واحد ===== */
function load_transfer(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM stock_transfers WHERE id=?");
  $h->execute([$id]); $header = $h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null, []];
  $i = $pdo->prepare("
    SELECT i.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM stock_transfer_items i
    JOIN product_variants v ON v.id=i.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE i.transfer_id=?
    ORDER BY i.id
  ");
  $i->execute([$id]); $items = $i->fetchAll(PDO::FETCH_ASSOC);
  return [$header, $items];
}
if (isset($_GET['getTransfer'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id = (int)$_GET['getTransfer'];
  [$h,$it] = load_transfer($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it], JSON_UNESCAPED_UNICODE);
  exit;
}

/* ===== تنفيذ المخزون للحركة (delta processed) ===== */
function apply_transfer_delta(PDO $pdo, int $transfer_id, int $from_branch, int $to_branch, array $prevProcByVar){
  // احسب المعالَج الحالي لكل variant
  $now = $pdo->prepare("SELECT variant_id, SUM(processed_qty) AS s FROM stock_transfer_items WHERE transfer_id=? GROUP BY variant_id");
  $now->execute([$transfer_id]);
  $nowMap = [];
  foreach ($now->fetchAll(PDO::FETCH_ASSOC) as $r) $nowMap[(int)$r['variant_id']] = (int)$r['s'];

  $pdo->beginTransaction();
  try {
    foreach ($nowMap as $vid=>$nowQty){
      $prev = (int)($prevProcByVar[$vid] ?? 0);
      $delta = $nowQty - $prev;
      if ($delta <= 0) continue;

      // تحقق رصيد المصدر
      $st = $pdo->prepare("SELECT quantity FROM product_variant_stock WHERE variant_id=? AND branch_id=? FOR UPDATE");
      $st->execute([$vid,$from_branch]);
      $srcQty = (int)($st->fetchColumn() ?: 0);
      if ($srcQty < $delta) throw new RuntimeException("كمية غير كافية في الفرع المرسل (Variant $vid): المطلوب $delta, المتاح $srcQty.");

      // خصم من المصدر
      $pdo->prepare("
        UPDATE product_variant_stock
        SET quantity = GREATEST(0, quantity - ?)
        WHERE variant_id=? AND branch_id=?
      ")->execute([$delta,$vid,$from_branch]);

      // إضافة للوجهة
      $pdo->prepare("
        INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
        VALUES (?,?,?,0)
        ON DUPLICATE KEY UPDATE quantity = quantity + VALUES(quantity)
      ")->execute([$vid,$to_branch,$delta]);

      // دفتر حركة (خارج/داخل)
      $uc = last_purchase_cost($pdo,$vid);
      $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                     VALUES (?,?,?,?, 'transfer_out', ?)")->execute([$vid,$from_branch,-$delta,$uc,$transfer_id]);
      $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                     VALUES (?,?,?,?, 'transfer_in', ?)")->execute([$vid,$to_branch,$delta,$uc,$transfer_id]);
    }
    $pdo->commit();
  } catch(Throwable $e){
    $pdo->rollBack();
    throw $e;
  }
}

/* للاحتفاظ بالمعالَج السابق قبل التحديث */
$_prev_processed_by_var = [];

/* ===== POST Actions ===== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).';
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية.';
  } else {
    $action = $_POST['action'] ?? '';
    try {

      if ($action === 'save_transfer') {
        $id            = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $from_branch   = (int)($_POST['from_branch_id'] ?? 0);
        $to_branch     = (int)($_POST['to_branch_id'] ?? 0);
        $status        = $_POST['status'] ?? 'draft';
        $transfer_date = $_POST['transfer_date'] ?: date('Y-m-d');
        $notes         = clean_str($_POST['notes'] ?? '', 10000);

        if ($from_branch<=0 || $to_branch<=0 || $from_branch===$to_branch) throw new RuntimeException('اختر فرعين مختلفين.');
        // بنود
        $var_ids  = array_map('intval', $_POST['item_variant_id'] ?? []);
        $desc_arr = $_POST['item_desc'] ?? [];
        $qty_arr  = array_map('intval', $_POST['item_qty'] ?? []);
        $proc_arr = array_map('intval', $_POST['item_processed_qty'] ?? []);
        $line_note_arr = $_POST['item_line_note'] ?? [];

        $items = [];
        for($i=0;$i<count($var_ids);$i++){
          if ($var_ids[$i] <= 0) continue;
          $qty  = max(0,(int)($qty_arr[$i] ?? 0));
          $proc = max(0,(int)($proc_arr[$i] ?? 0));
          if ($proc > $qty) $proc = $qty;
          $desc = clean_str($desc_arr[$i] ?? '', 255);
          $lnote= clean_str($line_note_arr[$i] ?? '', 255);
          if ($qty===0 && $proc===0) continue;
          $items[] = ['variant_id'=>$var_ids[$i],'description'=>$desc,'qty'=>$qty,'processed_qty'=>$proc,'line_note'=>$lnote];
        }
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // مرفق اختياري
        $attachName=null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error']===UPLOAD_ERR_OK){
          $dir = __DIR__ . '/uploads/transfers/';
          if (!is_dir($dir)) @mkdir($dir,0777,true);
          $ext = strtolower(pathinfo($_FILES['attachment']['name'], PATHINFO_EXTENSION));
          $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
          if (move_uploaded_file($_FILES['attachment']['tmp_name'], $dir.$name)) $attachName=$name;
        }

        if ($id){
          // إجمالي المعالَج السابق لكل variant
          $pr = $pdo->prepare("SELECT variant_id, SUM(processed_qty) s FROM stock_transfer_items WHERE transfer_id=? GROUP BY variant_id");
          $pr->execute([$id]);
          foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed_by_var[(int)$r['variant_id']] = (int)$r['s']; }

          // تحديث الهيدر
          $sql = "UPDATE stock_transfers SET from_branch_id=:fb, to_branch_id=:tb, status=:st,
                  transfer_date=:dt, notes=:notes".($attachName? ", attachment=:att": "")."
                  WHERE id=:id AND deleted_at IS NULL";
          $st = $pdo->prepare($sql);
          $p = [':fb'=>$from_branch, ':tb'=>$to_branch, ':st'=>$status, ':dt'=>$transfer_date, ':notes'=>$notes, ':id'=>$id];
          if ($attachName) $p[':att']=$attachName;
          $st->execute($p);

          // إعادة إدخال البنود
          $pdo->prepare("DELETE FROM stock_transfer_items WHERE transfer_id=?")->execute([$id]);
          $ins = $pdo->prepare("INSERT INTO stock_transfer_items (transfer_id, variant_id, description, qty, processed_qty, line_note)
                                VALUES (?,?,?,?,?,?)");
          foreach ($items as $it){
            $ins->execute([$id,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['line_note']]);
          }

          // تنفيذ مخزون عند partial/transferred (زيادة المعالَج)
          if (in_array($status,['partial','transferred'],true)) {
            apply_transfer_delta($pdo, $id, $from_branch, $to_branch, $_prev_processed_by_var);
          }

          $success = '✔️ تم تحديث التحويل';

        } else {
          // إنشاء هيدر
          $st = $pdo->prepare("INSERT INTO stock_transfers (from_branch_id, to_branch_id, status, transfer_date, notes, attachment)
                               VALUES (?,?,?,?,?,?)");
          $st->execute([$from_branch,$to_branch,$status,$transfer_date,$notes,$attachName]);
          $tid = (int)$pdo->lastInsertId();

          // إدخال البنود
          $ins = $pdo->prepare("INSERT INTO stock_transfer_items (transfer_id, variant_id, description, qty, processed_qty, line_note)
                                VALUES (?,?,?,?,?,?)");
          foreach ($items as $it){
            $ins->execute([$tid,$it['variant_id'],$it['description'],$it['qty'],$it['processed_qty'],$it['line_note']]);
          }

          // تنفيذ فوري لو partial/transferred
          if (in_array($status,['partial','transferred'],true)) {
            $_prev_processed_by_var = []; // جديد = 0
            apply_transfer_delta($pdo, $tid, $from_branch, $to_branch, $_prev_processed_by_var);
          }

          $success = '✔️ تم إنشاء التحويل';
        }

      } elseif ($action === 'execute_all') {
        $id = (int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('معرّف غير صالح.');
        // قراءة الهيدر
        $h = $pdo->prepare("SELECT from_branch_id, to_branch_id FROM stock_transfers WHERE id=?");
        $h->execute([$id]); $hdr = $h->fetch(PDO::FETCH_ASSOC);
        if (!$hdr) throw new RuntimeException('التحويل غير موجود.');
        // إجمالي المعالَج السابق لكل variant
        $pr = $pdo->prepare("SELECT variant_id, SUM(processed_qty) s FROM stock_transfer_items WHERE transfer_id=? GROUP BY variant_id");
        $pr->execute([$id]);
        foreach($pr->fetchAll(PDO::FETCH_ASSOC) as $r){ $_prev_processed_by_var[(int)$r['variant_id']] = (int)$r['s']; }

        // اجعل processed = qty لكل البنود
        $pdo->prepare("UPDATE stock_transfer_items SET processed_qty = qty WHERE transfer_id=?")->execute([$id]);
        // الحالة transferred
        $pdo->prepare("UPDATE stock_transfers SET status='transferred' WHERE id=?")->execute([$id]);

        // طبّق الفروق
        apply_transfer_delta($pdo, $id, (int)$hdr['from_branch_id'], (int)$hdr['to_branch_id'], $_prev_processed_by_var);
        $success = '✔️ تم تنفيذ التحويل بالكامل';

      } elseif ($action === 'soft_delete') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE stock_transfers SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success = 'تم أرشفة التحويل';

      } elseif ($action === 'restore') {
        $id = (int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE stock_transfers SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success = 'تم استعادة التحويل';
      }

    } catch(Throwable $e){
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ===== فلاتر العرض ===== */
$q            = trim($_GET['q'] ?? '');
$fFrom       = ($_GET['from_branch_id'] ?? '')!=='' ? (int)$_GET['from_branch_id'] : null;
$fTo         = ($_GET['to_branch_id'] ?? '')!=='' ? (int)$_GET['to_branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where=[]; $params=[];
$where[] = $showArchived ? "t.deleted_at IS NOT NULL OR t.deleted_at IS NULL" : "t.deleted_at IS NULL";
if ($q!==''){ $where[]="(t.id LIKE :q OR b1.`$BR_NAME_COL` LIKE :q OR b2.`$BR_NAME_COL` LIKE :q)"; $params[':q']="%$q%"; }
if ($fFrom){ $where[]="t.from_branch_id=:fb"; $params[':fb']=$fFrom; }
if ($fTo){ $where[]="t.to_branch_id=:tb"; $params[':tb']=$fTo; }
if ($fStatus!==''){ $where[]="t.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';

$sql = "
  SELECT t.*,
         b1.`$BR_NAME_COL` AS FromName,
         b2.`$BR_NAME_COL` AS ToName,
         COALESCE(SUM(i.qty),0) AS sum_qty,
         COALESCE(SUM(i.processed_qty),0) AS sum_proc
  FROM stock_transfers t
  LEFT JOIN stock_transfer_items i ON i.transfer_id=t.id
  LEFT JOIN `$BR_TBL` b1 ON b1.`$BR_ID_COL`=t.from_branch_id
  LEFT JOIN `$BR_TBL` b2 ON b2.`$BR_ID_COL`=t.to_branch_id
  $whereSql
  GROUP BY t.id
  ORDER BY t.id DESC
  LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>تحويلات المخزون</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .muted{color:var(--muted)}
    .table>:not(caption)>*>*{background:transparent}
    .num{min-width:90px;text-align:center}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= esc($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= esc($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-right-left me-2"></i> تحويلات المخزون</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">

  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث برقم التحويل/الفروع" value="<?= esc($q) ?>">
        <select class="form-select" name="from_branch_id" style="min-width:200px">
          <option value="">من كل الفروع</option>
          <?php foreach ($branches as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $fFrom===$bid?'selected':''; ?>><?= esc($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="to_branch_id" style="min-width:200px">
          <option value="">إلى كل الفروع</option>
          <?php foreach ($branches as $bid=>$bname): ?>
            <option value="<?= (int)$bid ?>" <?= $fTo===$bid?'selected':''; ?>><?= esc($bname) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:200px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','requested'=>'معتمدة','partial'=>'تنفيذ جزئي','transferred'=>'تم التحويل','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fFrom || $fTo || $fStatus || $showArchived): ?>
          <a href="transfers.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <?php if ($canEdit): ?>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#transferModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> تحويل جديد
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>من فرع</th>
              <th>إلى فرع</th>
              <th>الحالة</th>
              <th>إجمالي الكمية</th>
              <th>منفّذ</th>
              <th>متبقي</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r): 
              $sum = (int)$r['sum_qty']; $proc=(int)$r['sum_proc']; $due = max(0,$sum-$proc);
          ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td><?= esc($r['transfer_date']) ?></td>
              <td><span class="pill"><?= esc($r['FromName'] ?? '-') ?></span></td>
              <td><span class="pill"><?= esc($r['ToName'] ?? '-') ?></span></td>
              <td>
                <?php
                  $badges = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'requested'=>'<span class="badge bg-info text-dark">معتمدة</span>',
                    'partial'=>'<span class="badge bg-warning text-dark">تنفيذ جزئي</span>',
                    'transferred'=>'<span class="badge bg-success">تم التحويل</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>'
                  ];
                  echo $badges[$r['status']] ?? esc($r['status']);
                ?>
              </td>
              <td><?= $sum ?></td>
              <td><?= $proc ?></td>
              <td><?= $due ?></td>
              <td>
                <div class="btn-group">
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <?php if ($r['status']!=='transferred' && $r['status']!=='cancelled'): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('تنفيذ التحويل بالكامل؟ سيؤثر على المخزون.')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="execute_all">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-dolly"></i></button>
                    </form>
                  <?php endif; ?>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة التحويل؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Create/Edit Transfer -->
<div class="modal fade" id="transferModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-right-left me-2"></i> <span id="modalTitle">تحويل جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_transfer">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">من فرع *</label>
            <select class="form-select" name="from_branch_id" id="f_from" required>
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= esc($bname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">إلى فرع *</label>
            <select class="form-select" name="to_branch_id" id="f_to" required>
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bname): ?>
                <option value="<?= (int)$bid ?>"><?= esc($bname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="requested">معتمدة</option>
              <option value="partial">تنفيذ جزئي</option>
              <option value="transferred">تم التحويل</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>

          <div class="col-md-3">
            <label class="form-label">تاريخ التحويل</label>
            <input type="date" class="form-control" name="transfer_date" id="f_date" value="<?= date('Y-m-d') ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">مرفق (اختياري)</label>
            <input type="file" class="form-control" name="attachment" accept="image/*,.pdf">
          </div>
          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive">
              <table class="table table-sm align-middle text-center">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:240px">المتغيّر (SKU)</th>
                    <th>وصف</th>
                    <th class="num">الكمية</th>
                    <th class="num">منفّذ</th>
                    <th>ملاحظة</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- إجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-5">
                <div style="background:rgba(0,0,0,.03); border-radius:12px; padding:10px">
                  <div class="d-flex justify-content-between"><span>إجمالي الكمية:</span><strong id="t_sum">0</strong></div>
                  <div class="d-flex justify-content-between"><span>منفّذ:</span><strong id="t_proc">0</strong></div>
                  <hr class="my-2">
                  <div class="d-flex justify-content-between fs-5"><span>متبقي:</span><strong id="t_due">0</strong></div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>
    </div>
  </div>
</div>

<datalist id="variantsList">
  <?php foreach ($variants as $v): ?>
    <option value="<?= (int)$v['id'] ?> | <?= esc($v['label']) ?>"></option>
  <?php endforeach; ?>
</datalist>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent=d?'☀️':'🌙'; });

  const transferModal = new bootstrap.Modal(document.getElementById('transferModal'));
  const itemsBody = document.getElementById('itemsBody');

  function openCreate(){
    document.getElementById('modalTitle').textContent='تحويل جديد';
    ['f_id','f_notes'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=''; });
    document.getElementById('f_status').value='draft';
    document.getElementById('f_date').value = new Date().toISOString().slice(0,10);
    document.getElementById('f_from').value=''; document.getElementById('f_to').value='';
    itemsBody.innerHTML=''; addRow(); calcTotals();
  }

  function openEdit(id){
    fetch(location.href + (location.search ? '&':'?') + 'getTransfer=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items);
        transferModal.show();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل تحويل #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_from').value = h.from_branch_id||'';
    document.getElementById('f_to').value = h.to_branch_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_date').value = h.transfer_date||'';
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)" value="${it.variant_id} | ${escapeHtml(it.product_name||'')} | ${escapeHtml(it.sku||'')}">
        <input type="hidden" name="item_variant_id[]" value="${it.variant_id}">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control" value="${escapeHtml(it.description||'')}"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="${it.qty||0}" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="${it.processed_qty||0}" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="text" name="item_line_note[]" class="form-control" value="${escapeHtml(it.line_note||'')}"></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function addRow(){
    const tr = document.createElement('tr'); tr.className='item-row';
    tr.innerHTML = `
      <td>
        <input class="form-control" list="variantsList" placeholder="اكتب للبحث..." onblur="parseVariant(this)">
        <input type="hidden" name="item_variant_id[]">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control"></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="1" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="number" name="item_processed_qty[]" class="form-control text-center" value="0" oninput="clampProcessed(this.closest('tr')); calcTotals();"></td>
      <td><input type="text" name="item_line_note[]" class="form-control"></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function clampProcessed(tr){
    const qty = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const proc = tr.querySelector('[name="item_processed_qty[]"]');
    let v = parseFloat(proc.value||0); if (v>qty) v=qty; if (v<0) v=0; proc.value = v;
    calcTotals();
  }

  function parseVariant(inp){
    const hidden = inp.parentElement.querySelector('input[type=hidden]');
    const val = (inp.value||'').trim();
    const id = parseInt(val.split('|')[0],10);
    if (!isNaN(id)) hidden.value = id;
    calcTotals();
  }

  function calcTotals(){
    let sum=0, proc=0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      sum += parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
      proc += parseFloat(tr.querySelector('[name="item_processed_qty[]"]').value||0);
    });
    document.getElementById('t_sum').textContent = sum.toFixed(0);
    document.getElementById('t_proc').textContent = proc.toFixed(0);
    document.getElementById('t_due').textContent = (sum-proc).toFixed(0);
  }

  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
</script>
</body>
</html>
