<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ===== Helpers ===== */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function hasColumn(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
  $st->execute([$table,$col]); return (int)$st->fetchColumn() > 0;
}
function clean_str(?string $s, int $max=255): ?string{ $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }
function money($n){ return number_format((float)$n,2); }

/* ===== صلاحيات ===== */
$canEdit = true;

/* ===== CSRF ===== */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ===== كشف أعمدة الفروع / الفئات / الماركات ===== */
function detectCols(PDO $pdo, string $table, array $idCands, array $nameCands): array {
  $idCol = $idCands[0]; $nmCol = $nameCands[0];
  foreach ($idCands as $c)  if (hasColumn($pdo,$table,$c)) { $idCol=$c; break; }
  foreach ($nameCands as $c)if (hasColumn($pdo,$table,$c)) { $nmCol=$c; break; }
  return [$idCol,$nmCol];
}
$BR_TBL = 'branches';
[$BR_ID_COL, $BR_NAME_COL]       = detectCols($pdo,'branches',['id','branch_id','BranchID','ID'],['name','branch_name','BranchName','Name']);
[$CAT_ID_COL,$CAT_NAME_COL]       = detectCols($pdo,'categories',['id','CategoryID'],['name','CategoryName']);
[$BRAND_ID_COL,$BRAND_NAME_COL]   = detectCols($pdo,'brands',['id','BrandID'],['name','BrandName']);

/* ===== تأكيد جداول الجرد (إن لم توجد) + أعمدة إضافية اختيارية ===== */
$pdo->exec("
CREATE TABLE IF NOT EXISTS stock_counts (
  id INT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  status ENUM('draft','in_progress','finalized','cancelled') NOT NULL DEFAULT 'draft',
  count_date DATE NOT NULL,
  notes TEXT NULL,
  attachment VARCHAR(255) NULL,
  deleted_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS stock_count_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  count_id INT NOT NULL,
  variant_id INT NOT NULL,
  expected_qty INT NOT NULL DEFAULT 0,
  counted_qty INT NOT NULL DEFAULT 0,
  reason ENUM('','Damage','Theft','Write-off','Count Error') NOT NULL DEFAULT '',
  line_note VARCHAR(255) NULL,
  CONSTRAINT fk_sci_header FOREIGN KEY (count_id) REFERENCES stock_counts(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
/* لو الجدول قديم بدون عمود reason، أضِفه بأمان */
if (!hasColumn($pdo,'stock_count_items','reason')){
  $pdo->exec("ALTER TABLE stock_count_items ADD COLUMN reason ENUM('','Damage','Theft','Write-off','Count Error') NOT NULL DEFAULT ''");
}

/* ===== بيانات أساسية: فروع / فئات / ماركات ===== */
try {
  $branches = $pdo->query("SELECT `$BR_ID_COL` AS BranchID, `$BR_NAME_COL` AS BranchName FROM `$BR_TBL` ORDER BY `$BR_NAME_COL`")
                  ->fetchAll(PDO::FETCH_KEY_PAIR);
} catch(Throwable $e){ $branches = []; }

$categories = $brands = [];
try {
  if ($pdo->query("SHOW TABLES LIKE 'categories'")->fetchColumn()){
    $categories = $pdo->query("SELECT `$CAT_ID_COL` AS id, `$CAT_NAME_COL` AS name FROM `categories` ORDER BY `$CAT_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
  }
} catch(Throwable $e){}
try {
  if ($pdo->query("SHOW TABLES LIKE 'brands'")->fetchColumn()){
    $brands = $pdo->query("SELECT `$BRAND_ID_COL` AS id, `$BRAND_NAME_COL` AS name FROM `brands` ORDER BY `$BRAND_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
  }
} catch(Throwable $e){}

/* ===== آخر تكلفة شراء (للتسجيل) ===== */
function last_purchase_cost(PDO $pdo, int $variantId): float {
  try {
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]);
    $x=$st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  } catch(Throwable $e){ return 0.0; }
}

/* ===== Endpoints: Ajax ===== */

/* 1) تحميل جلسة واحدة JSON */
function load_count(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM stock_counts WHERE id=?");
  $h->execute([$id]); $header = $h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null,[]];
  $i = $pdo->prepare("
    SELECT i.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM stock_count_items i
    JOIN product_variants v ON v.id=i.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE i.count_id=?
    ORDER BY i.id
  ");
  $i->execute([$id]); $items = $i->fetchAll(PDO::FETCH_ASSOC);
  return [$header,$items];
}
if (isset($_GET['getCount'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id=(int)$_GET['getCount'];
  [$h,$it] = load_count($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it], JSON_UNESCAPED_UNICODE);
  exit;
}

/* 2) Prefill من المخزون مع فلاتر (فرع/فئة/ماركة/بحث/أقل من الحد الأدنى) */
if (isset($_GET['prefill']) && $_GET['prefill']=='1') {
  header('Content-Type: application/json; charset=utf-8');

  $bid   = (int)($_GET['branch_id'] ?? 0);
  $catId = ($_GET['category_id'] ?? '')!=='' ? (int)$_GET['category_id'] : null;
  $brId  = ($_GET['brand_id'] ?? '')!==''    ? (int)$_GET['brand_id']    : null;
  $qLike = trim($_GET['q'] ?? '');
  $onlyBelowMin = isset($_GET['only_below_min']);

  if ($bid<=0){ echo json_encode(['ok'=>false,'error'=>'branch_id required']); exit; }

  $where = ["s.branch_id=?"]; $params = [$bid];

  if ($catId && (hasColumn($pdo,'products','category_id') || hasColumn($pdo,'products','CategoryID'))){
    $pc = hasColumn($pdo,'products','category_id') ? 'category_id' : 'CategoryID';
    $where[] = "p.`$pc` = ?"; $params[] = $catId;
  }
  if ($brId && (hasColumn($pdo,'products','brand_id') || hasColumn($pdo,'products','BrandID'))){
    $bc = hasColumn($pdo,'products','brand_id') ? 'brand_id' : 'BrandID';
    $where[] = "p.`$bc` = ?"; $params[] = $brId;
  }
  if ($qLike !== ''){
    $where[] = "(v.sku LIKE ? OR p.name_ar LIKE ? OR IFNULL(p.name_en,'') LIKE ?)";
    $params[] = "%$qLike%"; $params[] = "%$qLike%"; $params[] = "%$qLike%";
  }
  if ($onlyBelowMin){
    $where[] = "COALESCE(s.quantity,0) < COALESCE(s.min_quantity,0)";
  }
  $whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';
  $sql = "
    SELECT s.variant_id, COALESCE(s.quantity,0) AS expected_qty,
           v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM product_variant_stock s
    JOIN product_variants v ON v.id=s.variant_id
    JOIN products p        ON p.id=v.product_id
    $whereSql
    ORDER BY p.name_ar, v.id
    LIMIT 5000
  ";
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  foreach ($rows as &$r){ $r['counted_qty'] = (int)$r['expected_qty']; }
  echo json_encode(['ok'=>true,'rows'=>$rows], JSON_UNESCAPED_UNICODE);
  exit;
}

/* 3) Autocomplete Ajax للمتغيّرات */
if (isset($_GET['searchVariants'])) {
  header('Content-Type: application/json; charset=utf-8');
  $q = trim($_GET['q'] ?? '');
  if ($q===''){ echo json_encode(['ok'=>true,'items'=>[]]); exit; }
  $st = $pdo->prepare("
    SELECT v.id, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS label
    FROM product_variants v
    JOIN products p ON p.id=v.product_id
    WHERE v.deleted_at IS NULL AND (v.sku LIKE ? OR p.name_ar LIKE ? OR IFNULL(p.name_en,'') LIKE ?)
    ORDER BY p.name_ar, v.id
    LIMIT 20
  ");
  $like = "%$q%"; $st->execute([$like,$like,$like]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  echo json_encode(['ok'=>true,'items'=>$rows], JSON_UNESCAPED_UNICODE);
  exit;
}

/* 4) جلب رصيد متغير واحد لفرع معيّن (للـSKU السريع) */
if (isset($_GET['variant_stock'])) {
  header('Content-Type: application/json; charset=utf-8');
  $vid = (int)($_GET['variant_id'] ?? 0);
  $bid = (int)($_GET['branch_id'] ?? 0);
  if ($vid<=0 || $bid<=0){ echo json_encode(['ok'=>false]); exit; }
  $st = $pdo->prepare("SELECT COALESCE(quantity,0) FROM product_variant_stock WHERE variant_id=? AND branch_id=?");
  $st->execute([$vid,$bid]); $qty = (int)($st->fetchColumn() ?: 0);
  echo json_encode(['ok'=>true,'expected_qty'=>$qty], JSON_UNESCAPED_UNICODE);
  exit;
}

/* 5) Export CSV لبنود الجلسة */
if (isset($_GET['export_items']) && $_GET['export_items']!=='') {
  $id = (int)$_GET['export_items'];
  $st = $pdo->prepare("
    SELECT i.variant_id, v.sku, p.name_ar, IFNULL(p.name_en,'-') AS name_en, i.expected_qty, i.counted_qty, i.reason, i.line_note
    FROM stock_count_items i
    JOIN product_variants v ON v.id=i.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE i.count_id=?
    ORDER BY p.name_ar, v.id
  ");
  $st->execute([$id]); $rows=$st->fetchAll(PDO::FETCH_ASSOC);
  header('Content-Type: text/csv; charset=utf-8');
  header('Content-Disposition: attachment; filename="stock_count_'.$id.'_items.csv"');
  $out=fopen('php://output','w');
  fputcsv($out, ['variant_id','sku','name_ar','name_en','expected_qty','counted_qty','reason','note']);
  foreach($rows as $r) fputcsv($out, [$r['variant_id'],$r['sku'],$r['name_ar'],$r['name_en'],$r['expected_qty'],$r['counted_qty'],$r['reason'],$r['line_note']]);
  fclose($out);
  exit;
}

/* 6) طباعة ورقة الجرد لجلسة محفوظة */
if (isset($_GET['print']) && $_GET['print']!=='') {
  $id = (int)$_GET['print'];
  $h = $pdo->prepare("SELECT c.*, b.`$BR_NAME_COL` AS BranchName FROM stock_counts c LEFT JOIN `$BR_TBL` b ON b.`$BR_ID_COL`=c.branch_id WHERE c.id=?");
  $h->execute([$id]); $hdr = $h->fetch(PDO::FETCH_ASSOC);
  if (!$hdr){ http_response_code(404); echo "Not found"; exit; }

  $st = $pdo->prepare("
    SELECT i.variant_id, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name,
           i.expected_qty, i.counted_qty
    FROM stock_count_items i
    JOIN product_variants v ON v.id=i.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE i.count_id=?
    ORDER BY p.name_ar, v.id
  ");
  $st->execute([$id]); $rows=$st->fetchAll(PDO::FETCH_ASSOC);
  ?><!doctype html><html lang="ar" dir="rtl"><head><meta charset="utf-8"><title>طباعة جرد #<?= (int)$id ?></title>
  <style>body{font-family:"Tajawal",Tahoma,Arial,sans-serif;margin:24px}h1,h2{margin:.2em 0}.muted{color:#666}
  table{width:100%;border-collapse:collapse;margin-top:12px}th,td{border:1px solid #ccc;padding:6px;text-align:center}
  th{background:#f4f6f8}@media print{.no-print{display:none}}</style></head><body>
  <div class="no-print" style="text-align:left;margin-bottom:8px"><button onclick="window.print()">🖨️ طباعة</button></div>
  <h2>ورقة جرد — جلسة #<?= (int)$id ?></h2>
  <div class="muted">الفرع: <strong><?= esc($hdr['BranchName']??'-') ?></strong> — التاريخ: <strong><?= esc($hdr['count_date']) ?></strong></div>
  <table><thead><tr><th>#</th><th>Variant ID</th><th>SKU</th><th>الاسم</th><th>متوقّع</th><th>معدود</th><th>توقيع</th><th>ملاحظة</th></tr></thead><tbody>
  <?php if(!$rows): ?><tr><td colspan="8" class="muted">لا توجد بنود.</td></tr><?php else: $i=1; foreach($rows as $r): ?>
  <tr><td><?= $i++ ?></td><td><?= (int)$r['variant_id'] ?></td><td><?= esc($r['sku']) ?></td>
      <td style="text-align:right"><?= esc($r['product_name']) ?></td>
      <td><?= (int)$r['expected_qty'] ?></td><td><?= (int)$r['counted_qty'] ?></td><td></td><td></td></tr>
  <?php endforeach; endif; ?></tbody></table></body></html><?php
  exit;
}

/* 7) طباعة ورقة جرد مباشرة من الفلاتر (بدون إنشاء جلسة) */
if (isset($_GET['print_sheet']) && $_GET['print_sheet']=='1') {
  $bid   = (int)($_GET['branch_id'] ?? 0);
  $catId = ($_GET['category_id'] ?? '')!=='' ? (int)$_GET['category_id'] : null;
  $brId  = ($_GET['brand_id'] ?? '')!==''    ? (int)$_GET['brand_id']    : null;
  $qLike = trim($_GET['q'] ?? '');
  $onlyBelowMin = isset($_GET['only_below_min']);
  if ($bid<=0){ http_response_code(400); echo "branch_id required"; exit; }

  $where = ["s.branch_id=?"]; $params = [$bid];
  if ($catId && (hasColumn($pdo,'products','category_id') || hasColumn($pdo,'products','CategoryID'))){
    $pc = hasColumn($pdo,'products','category_id') ? 'category_id' : 'CategoryID';
    $where[] = "p.`$pc` = ?"; $params[] = $catId;
  }
  if ($brId && (hasColumn($pdo,'products','brand_id') || hasColumn($pdo,'products','BrandID'))){
    $bc = hasColumn($pdo,'products','brand_id') ? 'brand_id' : 'BrandID';
    $where[] = "p.`$bc` = ?"; $params[] = $brId;
  }
  if ($qLike !== ''){
    $where[] = "(v.sku LIKE ? OR p.name_ar LIKE ? OR IFNULL(p.name_en,'') LIKE ?)";
    $params[] = "%$qLike%"; $params[] = "%$qLike%"; $params[] = "%$qLike%";
  }
  if ($onlyBelowMin){ $where[] = "COALESCE(s.quantity,0) < COALESCE(s.min_quantity,0)"; }
  $whereSql = $where ? ('WHERE '.implode(' AND ',$where)) : '';
  $sql = "
    SELECT s.variant_id, COALESCE(s.quantity,0) AS expected_qty,
           v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM product_variant_stock s
    JOIN product_variants v ON v.id=s.variant_id
    JOIN products p        ON p.id=v.product_id
    $whereSql
    ORDER BY p.name_ar, v.id
    LIMIT 5000
  ";
  $st = $pdo->prepare($sql); $st->execute($params); $rows = $st->fetchAll(PDO::FETCH_ASSOC);

  $bn = '';
  try { $bn = $pdo->prepare("SELECT `$BR_NAME_COL` FROM `$BR_TBL` WHERE `$BR_ID_COL`=?"); $bn->execute([$bid]); $bn = (string)$bn->fetchColumn(); } catch(Exception $e){}
  ?><!doctype html><html lang="ar" dir="rtl"><head><meta charset="utf-8"><title>ورقة جرد (مصفّاة)</title>
  <style>body{font-family:"Tajawal",Tahoma,Arial,sans-serif;margin:24px}h1,h2{margin:.2em 0}.muted{color:#666}
  table{width:100%;border-collapse:collapse;margin-top:12px}th,td{border:1px solid #ccc;padding:6px;text-align:center}
  th{background:#f4f6f8}@media print{.no-print{display:none}}</style></head><body>
  <div class="no-print" style="text-align:left;margin-bottom:8px"><button onclick="window.print()">🖨️ طباعة</button></div>
  <h2>ورقة جرد (مصفّاة)</h2>
  <div class="muted">الفرع: <strong><?= esc($bn?:'-') ?></strong> — التاريخ: <strong><?= date('Y-m-d') ?></strong></div>
  <table><thead><tr><th>#</th><th>Variant ID</th><th>SKU</th><th>الاسم</th><th>متوقّع</th><th>معدود</th><th>توقيع</th><th>ملاحظة</th></tr></thead><tbody>
  <?php if(!$rows): ?><tr><td colspan="8" class="muted">لا توجد بنود حسب الفلاتر.</td></tr><?php else: $i=1; foreach($rows as $r): ?>
  <tr><td><?= $i++ ?></td><td><?= (int)$r['variant_id'] ?></td><td><?= esc($r['sku']) ?></td>
      <td style="text-align:right"><?= esc($r['product_name']) ?></td>
      <td><?= (int)$r['expected_qty'] ?></td><td></td><td></td><td></td></tr>
  <?php endforeach; endif; ?></tbody></table></body></html><?php
  exit;
}

/* ===== اعتماد الجرد: يضبط الرصيد = counted_qty ويسجل فروق ===== */
function finalize_count(PDO $pdo, int $count_id, int $branch_id){
  $st = $pdo->prepare("SELECT variant_id, counted_qty FROM stock_count_items WHERE count_id=?");
  $st->execute([$count_id]);
  $items = $st->fetchAll(PDO::FETCH_ASSOC);
  if (!$items) throw new RuntimeException('لا توجد بنود للجرد.');

  $pdo->beginTransaction();
  try{
    foreach ($items as $it){
      $vid = (int)$it['variant_id'];
      $counted = max(0,(int)$it['counted_qty']);

      $sel = $pdo->prepare("SELECT quantity FROM product_variant_stock WHERE variant_id=? AND branch_id=? FOR UPDATE");
      $sel->execute([$vid,$branch_id]);
      $curRow = $sel->fetch(PDO::FETCH_NUM);
      $cur = $curRow ? (int)$curRow[0] : 0;

      $delta = $counted - $cur;

      if (!$curRow){
        $pdo->prepare("INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity) VALUES (?,?,?,0)")
            ->execute([$vid,$branch_id,$counted]);
      } else {
        $pdo->prepare("UPDATE product_variant_stock SET quantity=? WHERE variant_id=? AND branch_id=?")
            ->execute([$counted,$vid,$branch_id]);
      }

      if ($delta !== 0){
        $uc = last_purchase_cost($pdo,$vid);
        $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id, notes)
                       VALUES (?,?,?,?, 'stock_count', ?, 'daily stock count')")
            ->execute([$vid,$branch_id,$delta,$uc,$count_id]);
      }
    }
    $pdo->prepare("UPDATE stock_counts SET status='finalized' WHERE id=?")->execute([$count_id]);
    $pdo->commit();
  } catch(Throwable $e){
    $pdo->rollBack();
    throw $e;
  }
}

/* ===== POST Actions ===== */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error='طلب غير صالح (CSRF).';
  } elseif (!$canEdit) {
    $error='لا تملك صلاحية.';
  } else {
    $action = $_POST['action'] ?? '';
    try {

      if ($action==='save_count'){
        $id         = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $branch_id  = (int)($_POST['branch_id'] ?? 0);
        $status     = $_POST['status'] ?? 'draft';
        $count_date = $_POST['count_date'] ?: date('Y-m-d');
        $notes      = clean_str($_POST['notes'] ?? '', 10000);
        if ($branch_id<=0) throw new RuntimeException('اختر فرعًا.');

        $var_ids = array_map('intval', $_POST['item_variant_id'] ?? []);
        $exp_arr = array_map('intval', $_POST['item_expected_qty'] ?? []);
        $cnt_arr = array_map('intval', $_POST['item_counted_qty'] ?? []);
        $reason_arr = $_POST['item_reason'] ?? [];
        $note_arr= $_POST['item_line_note'] ?? [];

        $items=[];
        for($i=0;$i<count($var_ids);$i++){
          if ($var_ids[$i] <= 0) continue;
          $exp = max(0,(int)($exp_arr[$i] ?? 0));
          $cnt = max(0,(int)($cnt_arr[$i] ?? 0));
          $rsn = clean_str($reason_arr[$i] ?? '', 30);
          $ln  = clean_str($note_arr[$i] ?? '', 255);
          $items[] = ['variant_id'=>$var_ids[$i], 'expected_qty'=>$exp, 'counted_qty'=>$cnt, 'reason'=>$rsn, 'line_note'=>$ln];
        }
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // مرفق
        $attachName=null;
        if (isset($_FILES['attachment']) && $_FILES['attachment']['error']===UPLOAD_ERR_OK){
          $dir = __DIR__ . '/uploads/stock_counts/';
          if (!is_dir($dir)) @mkdir($dir,0777,true);
          $ext = strtolower(pathinfo($_FILES['attachment']['name'], PATHINFO_EXTENSION));
          $name = time().'_'.bin2hex(random_bytes(4)).'.'.$ext;
          if (move_uploaded_file($_FILES['attachment']['tmp_name'], $dir.$name)) $attachName=$name;
        }

        if ($id){
          $sql = "UPDATE stock_counts SET branch_id=:b, status=:st, count_date=:dt, notes=:notes"
               . ($attachName? ", attachment=:att":"")
               . " WHERE id=:id AND deleted_at IS NULL";
          $st = $pdo->prepare($sql);
          $p  = [':b'=>$branch_id, ':st'=>$status, ':dt'=>$count_date, ':notes'=>$notes, ':id'=>$id];
          if ($attachName) $p[':att']=$attachName;
          $st->execute($p);

          $pdo->prepare("DELETE FROM stock_count_items WHERE count_id=?")->execute([$id]);
          $ins=$pdo->prepare("INSERT INTO stock_count_items (count_id, variant_id, expected_qty, counted_qty, reason, line_note) VALUES (?,?,?,?,?,?)");
          foreach($items as $it) $ins->execute([$id,$it['variant_id'],$it['expected_qty'],$it['counted_qty'],$it['reason'],$it['line_note']]);

          if ($status==='finalized') finalize_count($pdo,$id,$branch_id);
          $success='✔️ تم تحديث الجرد';

        } else {
          $st=$pdo->prepare("INSERT INTO stock_counts (branch_id, status, count_date, notes, attachment) VALUES (?,?,?,?,?)");
          $st->execute([$branch_id,$status,$count_date,$notes,$attachName]);
          $cid = (int)$pdo->lastInsertId();

          $ins=$pdo->prepare("INSERT INTO stock_count_items (count_id, variant_id, expected_qty, counted_qty, reason, line_note) VALUES (?,?,?,?,?,?)");
          foreach($items as $it) $ins->execute([$cid,$it['variant_id'],$it['expected_qty'],$it['counted_qty'],$it['reason'],$it['line_note']]);

          if ($status==='finalized') finalize_count($pdo,$cid,$branch_id);
          $success='✔️ تم إنشاء الجرد';
        }

      } elseif ($action==='finalize_now'){
        $id=(int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('معرّف غير صالح.');
        $h=$pdo->prepare("SELECT branch_id FROM stock_counts WHERE id=?");
        $h->execute([$id]); $bid=(int)$h->fetchColumn();
        if ($bid<=0) throw new RuntimeException('فرع غير معروف.');
        finalize_count($pdo,$id,$bid);
        $success = '✔️ تم اعتماد الجرد وتحديث المخزون';

      } elseif ($action==='import_csv'){
        $id = (int)($_POST['id'] ?? 0);
        if ($id<=0) throw new RuntimeException('جلسة غير معروفة.');
        if (!isset($_FILES['csv']) || $_FILES['csv']['error']!==UPLOAD_ERR_OK) throw new RuntimeException('ارفع ملف CSV صالح.');

        $tmp = $_FILES['csv']['tmp_name'];
        $h = fopen($tmp,'r'); if (!$h) throw new RuntimeException('تعذر قراءة الملف.');

        $upd = $pdo->prepare("UPDATE stock_count_items SET counted_qty=?, reason=COALESCE(?,reason) WHERE count_id=? AND variant_id=?");
        $updated = 0;

        // حاول تخطي هيدر
        $first = fgetcsv($h);
        if ($first && isset($first[0]) && !is_numeric($first[0])) {
          // تجاهل الهيدر
        } else {
          if ($first && count($first)>=2){
            $vid = (int)$first[0]; $cnt = (int)$first[1]; $rsn = $first[2] ?? null;
            $upd->execute([$cnt, $rsn, $id, $vid]); $updated += $upd->rowCount();
          }
        }
        while(($r=fgetcsv($h))!==false){
          if (count($r)<2) continue;
          $vid = (int)$r[0]; $cnt = (int)$r[1]; $rsn = $r[2] ?? null;
          $upd->execute([$cnt, $rsn, $id, $vid]); $updated += $upd->rowCount();
        }
        fclose($h);
        $success = "✔️ تم تحديث $updated بند من CSV";

      } elseif ($action==='soft_delete'){
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE stock_counts SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success='تم أرشفة الجلسة';

      } elseif ($action==='restore'){
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE stock_counts SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success='تم استعادة الجلسة';
      }

    } catch(Throwable $e){
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ===== فلاتر قائمة الجلسات ===== */
$q            = trim($_GET['q'] ?? '');
$fBranch      = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where=[]; $params=[];
$where[] = $showArchived ? "c.deleted_at IS NOT NULL OR c.deleted_at IS NULL" : "c.deleted_at IS NULL";
if ($q!==''){ $where[]="(c.id LIKE :q OR b.`$BR_NAME_COL` LIKE :q)"; $params[':q']="%$q%"; }
if ($fBranch){ $where[]="c.branch_id=:bid"; $params[':bid']=$fBranch; }
if ($fStatus!==''){ $where[]="c.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

$sql = "
  SELECT c.*, b.`$BR_NAME_COL` AS BranchName,
         COUNT(i.id) AS items_count,
         COALESCE(SUM(i.counted_qty - i.expected_qty),0) AS sum_delta
  FROM stock_counts c
  LEFT JOIN stock_count_items i ON i.count_id=c.id
  LEFT JOIN `$BR_TBL` b ON b.`$BR_ID_COL`=c.branch_id
  $whereSql
  GROUP BY c.id
  ORDER BY c.id DESC
  LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>الجرد اليومي</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .table>:not(caption)>*>*{background:transparent}
    .num{min-width:90px;text-align:center}
    .soft{background:rgba(0,0,0,.03);border-radius:10px;padding:6px 10px}
    .neg{background:#f8d7da;border:1px solid #f5c2c7;border-radius:999px;padding:2px 8px}
    .pos{background:#d1e7dd;border:1px solid #badbcc;border-radius:999px;padding:2px 8px}
    .badge-muted{background:rgba(0,0,0,.06); color:#333}
    .typeahead{position:absolute; z-index:9999; background:var(--card); border:1px solid var(--border); border-radius:8px; width:100%; max-height:240px; overflow:auto; display:none;}
    .typeahead div{padding:6px 10px; cursor:pointer}
    .typeahead div:hover{background:rgba(0,0,0,.05)}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= esc($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= esc($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-clipboard-check me-2"></i> الجرد اليومي</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث برقم الجرد/اسم الفرع" value="<?= esc($q) ?>">
        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>" <?= $fBranch===$bid?'selected':''; ?>><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:200px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','in_progress'=>'قيد الجرد','finalized'=>'معتمد','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fBranch || $fStatus || $showArchived): ?>
          <a href="stock_count.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <?php if ($canEdit): ?>
      <div class="d-flex gap-2">
        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#countModal" onclick="openCreate()">
          <i class="fa-solid fa-plus"></i> جلسة جرد
        </button>
        <!-- طباعة ورقة جرد مباشرة من الفلاتر -->
        <div class="dropdown">
          <button class="btn btn-outline-dark dropdown-toggle" data-bs-toggle="dropdown"><i class="fa-solid fa-print"></i> طباعة ورقة</button>
          <div class="dropdown-menu p-3" style="min-width:360px">
            <div class="mb-2">
              <label class="form-label">فرع</label>
              <select class="form-select" id="ps_branch">
                <?php foreach ($branches as $bid=>$bn): ?><option value="<?= (int)$bid ?>"><?= esc($bn) ?></option><?php endforeach; ?>
              </select>
            </div>
            <div class="row g-2">
              <div class="col-6">
                <label class="form-label">فئة</label>
                <select class="form-select" id="ps_cat">
                  <option value="">كل الفئات</option>
                  <?php foreach ($categories as $cid=>$cname): ?><option value="<?= (int)$cid ?>"><?= esc($cname) ?></option><?php endforeach; ?>
                </select>
              </div>
              <div class="col-6">
                <label class="form-label">ماركة</label>
                <select class="form-select" id="ps_brand">
                  <option value="">كل الماركات</option>
                  <?php foreach ($brands as $bid=>$bname): ?><option value="<?= (int)$bid ?>"><?= esc($bname) ?></option><?php endforeach; ?>
                </select>
              </div>
            </div>
            <div class="mt-2">
              <label class="form-label">بحث</label>
              <input type="text" class="form-control" id="ps_q" placeholder="SKU/اسم">
            </div>
            <div class="form-check my-2">
              <input class="form-check-input" type="checkbox" id="ps_below_min">
              <label class="form-check-label" for="ps_below_min">فقط الأقل من الحد الأدنى</label>
            </div>
            <a class="btn btn-dark w-100" id="ps_print" target="_blank"><i class="fa-solid fa-print"></i> طباعة</a>
          </div>
        </div>
      </div>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول الجلسات -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>الفرع</th>
              <th>الحالة</th>
              <th>عدد البنود</th>
              <th>صافي الفارق</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="7" class="text-muted">لا توجد جلسات.</td></tr>
          <?php else: foreach ($rows as $r):
              $delta = (int)$r['sum_delta'];
          ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td><?= esc($r['count_date']) ?></td>
              <td><span class="pill"><?= esc($r['BranchName'] ?? '-') ?></span></td>
              <td>
                <?php
                  $badges = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'in_progress'=>'<span class="badge bg-info text-dark">قيد الجرد</span>',
                    'finalized'=>'<span class="badge bg-success">معتمد</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>',
                  ];
                  echo $badges[$r['status']] ?? esc($r['status']);
                ?>
              </td>
              <td><?= (int)$r['items_count'] ?></td>
              <td><?php if ($delta>0): ?><span class="pos">+<?= $delta ?></span><?php elseif($delta<0): ?><span class="neg"><?= $delta ?></span><?php else: ?>0<?php endif; ?></td>
              <td>
                <div class="btn-group">
                  <a class="btn btn-sm btn-outline-success" href="?export_items=<?= (int)$r['id'] ?>" title="تصدير CSV"><i class="fa-solid fa-file-csv"></i></a>
                  <a class="btn btn-sm btn-outline-dark" href="?print=<?= (int)$r['id'] ?>" target="_blank" title="طباعة"><i class="fa-solid fa-print"></i></a>
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <?php if ($r['status']!=='finalized' && $r['status']!=='cancelled'): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('اعتماد الجرد وتحديث المخزون؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="finalize_now">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-primary"><i class="fa-solid fa-clipboard-check"></i></button>
                    </form>
                  <?php endif; ?>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة الجلسة؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<!-- Modal: Create/Edit Count -->
<div class="modal fade" id="countModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-list-check me-2"></i> <span id="modalTitle">جلسة جرد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <form method="post" enctype="multipart/form-data" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_count">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-4">
            <label class="form-label">الفرع *</label>
            <select class="form-select" name="branch_id" id="f_branch" required>
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bn): ?>
                <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="in_progress">قيد الجرد</option>
              <option value="finalized">معتمد</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">تاريخ الجرد</label>
            <input type="date" class="form-control" name="count_date" id="f_date" value="<?= date('Y-m-d') ?>">
          </div>

          <div class="col-md-3">
            <label class="form-label">مرفق (اختياري)</label>
            <input type="file" class="form-control" name="attachment" accept="image/*,.pdf">
          </div>
          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- فلاتر الجرد الجزئي + أزرار -->
          <div class="col-12">
            <div class="row g-2 align-items-end">
              <div class="col-md-3">
                <label class="form-label">فئة (اختياري)</label>
                <select class="form-select" id="pref_cat">
                  <option value="">كل الفئات</option>
                  <?php foreach ($categories as $cid=>$cname): ?>
                    <option value="<?= (int)$cid ?>"><?= esc($cname) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="col-md-3">
                <label class="form-label">ماركة (اختياري)</label>
                <select class="form-select" id="pref_brand">
                  <option value="">كل الماركات</option>
                  <?php foreach ($brands as $bid=>$bname): ?>
                    <option value="<?= (int)$bid ?>"><?= esc($bname) ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="col-md-3">
                <label class="form-label">بحث SKU / اسم</label>
                <input type="text" class="form-control" id="pref_q" placeholder="مثال: ABC أو تيشيرت">
              </div>
              <div class="col-md-3">
                <div class="form-check mt-4">
                  <input class="form-check-input" type="checkbox" id="pref_below_min">
                  <label class="form-check-label" for="pref_below_min">فقط الأقل من الحد الأدنى</label>
                </div>
              </div>

              <div class="col-12 d-flex flex-wrap gap-2 mt-1">
                <button type="button" class="btn btn-outline-secondary" onclick="prefillFromStock()">
                  <i class="fa-solid fa-download"></i> تحميل الرصيد (مصفّى)
                </button>

                <a class="btn btn-outline-success" id="exportBtn" href="#" target="_blank">
                  <i class="fa-solid fa-file-csv"></i> تصدير CSV
                </a>

                <!-- استيراد CSV -->
                <label class="btn btn-outline-primary mb-0">
                  <i class="fa-solid fa-file-arrow-up"></i> استيراد CSV
                  <input type="file" name="csv" id="csvInput" accept=".csv" hidden>
                </label>

                <a id="printBtn" class="btn btn-outline-dark" href="#" target="_blank">
                  <i class="fa-solid fa-print"></i> طباعة ورقة الجرد
                </a>

                <!-- إدخال SKU سريع يزوّد المعدود -->
                <div class="ms-auto d-flex align-items-center gap-2">
                  <span class="badge badge-muted">إدخال سريع بالـSKU</span>
                  <input type="text" id="scanSku" class="form-control" placeholder="اكتب SKU واضغط Enter" style="min-width:220px">
                </div>
              </div>
            </div>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive position-relative">
              <table class="table table-sm align-middle text-center">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:280px">المتغيّر (بحث حي)</th>
                    <th class="num">متوقّع</th>
                    <th class="num">معدود</th>
                    <th class="num">فارق</th>
                    <th style="min-width:150px">سبب الفارق</th>
                    <th style="min-width:200px">ملاحظة</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
              <div id="typeaheadContainer" class="typeahead"></div>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- إجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-5">
                <div class="soft">
                  <div class="d-flex justify-content-between"><span>عدد البنود:</span><strong id="t_items">0</strong></div>
                  <div class="d-flex justify-content-between"><span>صافي الفارق:</span><strong id="t_delta">0</strong></div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>

      <!-- فورم استيراد CSV مخفي -->
      <form method="post" enctype="multipart/form-data" id="csvForm" class="d-none">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="import_csv">
        <input type="hidden" name="id" id="csv_count_id">
        <input type="file" name="csv" id="csv_file_hidden">
      </form>
    </div>
  </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent=d?'☀️':'🌙'; });

  // طباعة من الفلاتر في الهيدر
  document.getElementById('ps_print')?.addEventListener('click', (e)=>{
    const b = document.getElementById('ps_branch').value;
    const c = document.getElementById('ps_cat').value;
    const r = document.getElementById('ps_brand').value;
    const q = document.getElementById('ps_q').value;
    const below = document.getElementById('ps_below_min').checked ? '1':'';
    let url = new URL(location.href);
    url.searchParams.set('print_sheet','1');
    url.searchParams.set('branch_id', b);
    if (c) url.searchParams.set('category_id', c);
    if (r) url.searchParams.set('brand_id', r);
    if (q) url.searchParams.set('q', q);
    if (below) url.searchParams.set('only_below_min', '1');
    e.target.href = url.toString();
  });

  const countModal = new bootstrap.Modal(document.getElementById('countModal'));
  const itemsBody  = document.getElementById('itemsBody');
  const exportBtn  = document.getElementById('exportBtn');
  const printBtn   = document.getElementById('printBtn');
  const csvInput   = document.getElementById('csvInput');
  const typeahead  = document.getElementById('typeaheadContainer');
  let lastFocusedVariantInput = null;

  function openCreate(){
    document.getElementById('modalTitle').textContent='جلسة جرد جديدة';
    ['f_id','f_notes'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value=''; });
    document.getElementById('f_status').value='draft';
    document.getElementById('f_date').value = new Date().toISOString().slice(0,10);
    document.getElementById('f_branch').value='';
    itemsBody.innerHTML=''; addRow(); calcTotals();
    exportBtn.href = '#'; printBtn.href='#';
  }

  function openEdit(id){
    fetch(location.href + (location.search ? '&':'?') + 'getCount=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items); countModal.show(); syncLinks();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل جلسة #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_branch').value = h.branch_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_date').value = h.count_date||'';
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
    exportBtn.href = '?export_items=' + encodeURIComponent(h.id||'');
    printBtn.href  = '?print=' + encodeURIComponent(h.id||'');
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    const exp = +((it.expected_qty ?? 0) || 0);
    const cnt = +((it.counted_qty ?? 0) || 0);
    const delta = cnt - exp;
    tr.innerHTML = `
      <td class="text-start">
        <div class="position-relative">
          <input class="form-control variant-input" placeholder="اكتب SKU/اسم للبحث..." value="${it.variant_id ? (it.variant_id + ' | ' + escapeHtml(it.product_name||'') + (it.sku?(' | '+escapeHtml(it.sku)):'') ) : ''}" oninput="variantTypeahead(this)" onfocus="storeFocus(this)" onblur="hideTypeaheadDelayed()">
          <input type="hidden" name="item_variant_id[]" value="${it.variant_id||''}">
        </div>
      </td>
      <td><input type="number" name="item_expected_qty[]" class="form-control text-center" value="${exp}" oninput="updateDelta(this.closest('tr'))"></td>
      <td><input type="number" name="item_counted_qty[]"  class="form-control text-center" value="${cnt}" oninput="updateDelta(this.closest('tr'))"></td>
      <td class="num"><span class="pill line-delta">${delta}</span></td>
      <td>
        <select name="item_reason[]" class="form-select">
          ${['','Damage','Theft','Write-off','Count Error'].map(x=>`<option value="${x}" ${(it.reason||'')===x?'selected':''}>${x||'—'}</option>`).join('')}
        </select>
      </td>
      <td><input type="text" name="item_line_note[]" class="form-control" value="${escapeHtml(it.line_note||'')}"></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
  }

  function addRow(){
    addRowFromItem({variant_id:'', expected_qty:0, counted_qty:0, reason:'', line_note:''});
  }

  function updateDelta(tr){
    const exp = parseFloat(tr.querySelector('[name="item_expected_qty[]"]').value||0);
    const cnt = parseFloat(tr.querySelector('[name="item_counted_qty[]"]').value||0);
    tr.querySelector('.line-delta').textContent = (cnt-exp).toFixed(0);
    calcTotals();
  }

  function calcTotals(){
    let items=0, delta=0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      const vid = tr.querySelector('[name="item_variant_id[]"]').value;
      const exp = parseFloat(tr.querySelector('[name="item_expected_qty[]"]').value||0);
      const cnt = parseFloat(tr.querySelector('[name="item_counted_qty[]"]').value||0);
      if (vid) items++;
      delta += (cnt-exp);
    });
    document.getElementById('t_items').textContent = items.toFixed(0);
    document.getElementById('t_delta').textContent = delta.toFixed(0);
  }

  // Prefill
  async function prefillFromStock(){
    const bid = document.getElementById('f_branch').value;
    if (!bid){ alert('اختر الفرع أولًا.'); return; }

    const cat = document.getElementById('pref_cat')?.value || '';
    const br  = document.getElementById('pref_brand')?.value || '';
    const q   = document.getElementById('pref_q')?.value || '';
    const below = document.getElementById('pref_below_min')?.checked ? '1' : '';

    try{
      const url = new URL(location.href);
      url.searchParams.set('prefill','1');
      url.searchParams.set('branch_id', bid);
      if (cat)  url.searchParams.set('category_id', cat);
      if (br)   url.searchParams.set('brand_id', br);
      if (q)    url.searchParams.set('q', q);
      if (below)url.searchParams.set('only_below_min', '1');

      const res = await fetch(url.toString(), {headers:{'Accept':'application/json'}});
      const data = await res.json();
      if (!data.ok) throw new Error('failed');
      itemsBody.innerHTML='';
      (data.rows||[]).forEach(r=>{
        addRowFromItem({ variant_id:r.variant_id, expected_qty:r.expected_qty, counted_qty:r.counted_qty, product_name:r.product_name, sku:r.sku, reason:'', line_note:'' });
      });
      if (!(data.rows||[]).length) addRow();
      calcTotals();
    } catch(e){ alert('تعذر تحميل الرصيد.'); }
  }

  function syncLinks(){
    const id = document.getElementById('f_id').value||'';
    exportBtn.href = id? ('?export_items='+encodeURIComponent(id)) : '#';
    printBtn.href  = id? ('?print='+encodeURIComponent(id)) : '#';
  }
  document.getElementById('f_id')?.addEventListener('input', syncLinks);

  // استيراد CSV
  csvInput?.addEventListener('change', ()=>{
    const file = csvInput.files[0]; if (!file) return;
    const cid = document.getElementById('f_id').value||'';
    if (!cid){ alert('احفظ الجلسة أولًا ثم استورد CSV.'); csvInput.value=''; return; }
    const form = document.getElementById('csvForm');
    document.getElementById('csv_count_id').value = cid;
    document.getElementById('csv_file_hidden').files = csvInput.files;
    form.submit();
  });

  // ===== Autocomplete Ajax (Typeahead بسيط) =====
  let typeaheadTimer = null;
  function storeFocus(inp){ lastFocusedVariantInput = inp; }
  function hideTypeahead(){ typeahead.style.display='none'; typeahead.innerHTML=''; }
  function hideTypeaheadDelayed(){ setTimeout(hideTypeahead, 150); }

  async function variantTypeahead(inp){
    storeFocus(inp);
    const q = inp.value.trim();
    if (q.length < 2){ hideTypeahead(); return; }
    clearTimeout(typeaheadTimer);
    typeaheadTimer = setTimeout(async ()=>{
      try{
        const url = new URL(location.href);
        url.searchParams.set('searchVariants','1'); url.searchParams.set('q', q);
        const res = await fetch(url.toString(), {headers:{'Accept':'application/json'}});
        const data = await res.json();
        const rect = inp.getBoundingClientRect();
        const contRect = inp.closest('.table-responsive').getBoundingClientRect();
        typeahead.style.left = (rect.left - contRect.left) + 'px';
        typeahead.style.top  = (rect.bottom - contRect.top + inp.closest('.table-responsive').scrollTop) + 'px';
        typeahead.style.width = rect.width + 'px';
        typeahead.innerHTML='';
        (data.items||[]).forEach(it=>{
          const div = document.createElement('div');
          div.textContent = `${it.id} | ${it.label} | ${it.sku}`;
          div.addEventListener('click', ()=> selectVariant(inp, it));
          typeahead.appendChild(div);
        });
        typeahead.style.display = (typeahead.childElementCount>0) ? 'block':'none';
      } catch(e){ hideTypeahead(); }
    }, 200);
  }

  function selectVariant(inp, it){
    const hidden = inp.parentElement.querySelector('input[type=hidden]');
    inp.value = `${it.id} | ${it.label} | ${it.sku}`;
    hidden.value = it.id;
    hideTypeahead();
    // لو expected = 0، نحاول نجيب الرصيد الحالي
    const tr = inp.closest('tr');
    const expEl = tr.querySelector('[name="item_expected_qty[]"]');
    const bid = document.getElementById('f_branch').value||'';
    if (bid){
      fetch(location.href + (location.search ? '&':'?') + 'variant_stock=1&variant_id='+encodeURIComponent(it.id)+'&branch_id='+encodeURIComponent(bid))
        .then(r=>r.json()).then(d=>{ if (d.ok){ expEl.value = d.expected_qty||0; updateDelta(tr);} });
    } else { updateDelta(tr); }
  }

  // ===== إدخال سريع بالـSKU =====
  document.getElementById('scanSku')?.addEventListener('keydown', async (e)=>{
    if (e.key !== 'Enter') return;
    e.preventDefault();
    const sku = e.target.value.trim(); if (!sku) return;
    const res = await fetch(location.href + (location.search ? '&':'?') + 'searchVariants=1&q='+encodeURIComponent(sku));
    const data = await res.json();
    const match = (data.items||[]).find(x=> (x.sku||'').toLowerCase() === sku.toLowerCase()) || (data.items||[])[0];
    if (!match){ alert('SKU غير موجود'); return; }
    // لو الصف موجود: زوّد المعدود +1، غير كده أضف صف جديد
    const existing = [...document.querySelectorAll('#itemsBody .item-row')].find(tr=>{
      return tr.querySelector('[name="item_variant_id[]"]').value == match.id;
    });
    if (existing){
      const cntEl = existing.querySelector('[name="item_counted_qty[]"]');
      cntEl.value = (parseInt(cntEl.value||0)+1); updateDelta(existing);
    } else {
      addRowFromItem({variant_id:match.id, expected_qty:0, counted_qty:1, product_name:match.label, sku:match.sku, reason:'', line_note:''});
      // حمّل المتوقع من المخزون للفرع
      const bid = document.getElementById('f_branch').value||'';
      const tr = itemsBody.lastElementChild;
      const expEl = tr.querySelector('[name="item_expected_qty[]"]');
      if (bid){
        fetch(location.href + (location.search ? '&':'?') + 'variant_stock=1&variant_id='+encodeURIComponent(match.id)+'&branch_id='+encodeURIComponent(bid))
          .then(r=>r.json()).then(d=>{ if (d.ok){ expEl.value = d.expected_qty||0; updateDelta(tr);} });
      } else { updateDelta(tr); }
      // عيّن قيمة حقل البحث بعرض وصفي
      tr.querySelector('.variant-input').value = `${match.id} | ${match.label} | ${match.sku}`;
      tr.querySelector('[name="item_variant_id[]"]').value = match.id;
    }
    e.target.value = '';
    calcTotals();
  });

  function syncPrintSheetLink(){ /* dummy for completeness */ }

  // Print/Export links sync
  function syncLinks(){
    const id = document.getElementById('f_id').value||'';
    exportBtn.href = id? ('?export_items='+encodeURIComponent(id)) : '#';
    printBtn.href  = id? ('?print='+encodeURIComponent(id)) : '#';
  }

  // Utils
  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
</script>
</body>
</html>
