<?php
require_once __DIR__ . '/../core/auth.php';
require_login();
$pdo = db();

/* ============== CSRF ============== */
if (empty($_SESSION['csrf'])) {
  $_SESSION['csrf'] = bin2hex(random_bytes(16));
}
$CSRF = $_SESSION['csrf'];

/* ============== Helpers ============== */
function json_out($data, $code=200){
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE);
  exit;
}
function row_defaults(int $bid): array {
  return [
    'BranchID'             => $bid,
    'TaxRate'              => 14.00,
    'ServicePercent'       => 0.00,
    'Currency'             => 'EGP',
    'PrintFooter'          => null,
    'TaxInclusive'         => 1,
    'RoundingIncrement'    => 0.05,
    'PriceSource'          => 'branch',
    'AllowNegativeSales'   => 1,
    'DefaultPaymentMethod' => 'cash',
    'ReceiptWidthMM'       => 80,
    'ReceiptHeader'        => null,
    'DeliveryFeeMode'      => 'by_zone',
    'DeliveryFlatFee'      => 0.00,
  ];
}

/* ============== AJAX API ==============
   GET  ?action=get&branch_id=ID   => جلب إعدادات فرع
   POST JSON action=save           => حفظ إعدادات فرع
======================================= */
if (isset($_GET['action']) && $_GET['action']==='get') {
  $bid = (int)($_GET['branch_id'] ?? 0);

  // اسم الفرع والتحقق أنه نشط
  $stmt = $pdo->prepare("SELECT BranchName FROM branches WHERE BranchID=? AND IsActive=1");
  $stmt->execute([$bid]);
  $branchRow = $stmt->fetch(PDO::FETCH_ASSOC);
  if (!$branchRow) json_out(['ok'=>false,'msg'=>'فرع غير موجود أو غير نشط'], 404);

  // هات كل الأعمدة (أسلم في حالة إضافة حقول مستقبلًا)
  $st = $pdo->prepare("SELECT * FROM branchsettings WHERE BranchID=?");
  $st->execute([$bid]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  $data = row_defaults($bid);
  if ($row) {
    foreach ($data as $k=>$v) {
      if (array_key_exists($k,$row) && $row[$k] !== null) $data[$k] = $row[$k];
    }
  }
  $data['BranchName'] = $branchRow['BranchName'];
  json_out(['ok'=>true,'data'=>$data]);
}

if (isset($_POST['action']) && $_POST['action']==='save') {
  $payload = json_decode(file_get_contents('php://input'), true) ?: [];
  if (($payload['csrf'] ?? '') !== ($_SESSION['csrf'] ?? '')) {
    json_out(['ok'=>false,'msg'=>'CSRF غير صالح'], 403);
  }

  $BranchID       = (int)($payload['BranchID'] ?? 0);
  $TaxRate        = isset($payload['TaxRate']) ? (float)$payload['TaxRate'] : 0.0;
  $ServicePercent = isset($payload['ServicePercent']) ? (float)$payload['ServicePercent'] : 0.0;
  $Currency       = trim((string)($payload['Currency'] ?? 'EGP'));
  $PrintFooter    = trim((string)($payload['PrintFooter'] ?? ''));

  $TaxInclusive         = (int)($payload['TaxInclusive'] ?? 1) ? 1 : 0;
  $RoundingIncrement    = isset($payload['RoundingIncrement']) ? max(0.000, (float)$payload['RoundingIncrement']) : 0.05;
  $PriceSource          = in_array(($payload['PriceSource'] ?? 'branch'), ['branch','global'], true) ? $payload['PriceSource'] : 'branch';
  $AllowNegativeSales   = (int)($payload['AllowNegativeSales'] ?? 1) ? 1 : 0;
  $DefaultPaymentMethod = in_array(($payload['DefaultPaymentMethod'] ?? 'cash'), ['cash','card','wallet','split'], true) ? $payload['DefaultPaymentMethod'] : 'cash';
  $ReceiptWidthMM       = max(58, (int)($payload['ReceiptWidthMM'] ?? 80)); // أقل عرض 58مم
  $ReceiptHeader        = trim((string)($payload['ReceiptHeader'] ?? ''));
  $DeliveryFeeMode      = in_array(($payload['DeliveryFeeMode'] ?? 'by_zone'), ['none','flat','by_zone'], true) ? $payload['DeliveryFeeMode'] : 'by_zone';
  $DeliveryFlatFee      = max(0.00, (float)($payload['DeliveryFlatFee'] ?? 0.00));

  // قيود بسيطة
  if ($BranchID <= 0) json_out(['ok'=>false,'msg'=>'فرع غير محدد'], 422);
  if ($TaxRate < 0) $TaxRate = 0;
  if ($TaxRate > 100) $TaxRate = 100;
  if ($ServicePercent < 0) $ServicePercent = 0;
  if ($ServicePercent > 100) $ServicePercent = 100;
  if ($Currency === '') $Currency = 'EGP';
  if ($PrintFooter === '') $PrintFooter = null;
  if ($ReceiptHeader === '') $ReceiptHeader = null;

  // تأكد الفرع موجود ونشط
  $chk = $pdo->prepare("SELECT 1 FROM branches WHERE BranchID=? AND IsActive=1");
  $chk->execute([$BranchID]);
  if (!$chk->fetchColumn()) json_out(['ok'=>false,'msg'=>'الفرع غير موجود أو غير نشط'], 404);

  // UPSERT على branchsettings (lowercase)
  $exists = $pdo->prepare("SELECT 1 FROM branchsettings WHERE BranchID=?");
  $exists->execute([$BranchID]);

  if ($exists->fetchColumn()) {
    $q = "UPDATE branchsettings
          SET TaxRate=?, ServicePercent=?, Currency=?, PrintFooter=?,
              TaxInclusive=?, RoundingIncrement=?, PriceSource=?, AllowNegativeSales=?,
              DefaultPaymentMethod=?, ReceiptWidthMM=?, ReceiptHeader=?,
              DeliveryFeeMode=?, DeliveryFlatFee=?
          WHERE BranchID=?";
    $pdo->prepare($q)->execute([
      $TaxRate, $ServicePercent, $Currency, $PrintFooter,
      $TaxInclusive, $RoundingIncrement, $PriceSource, $AllowNegativeSales,
      $DefaultPaymentMethod, $ReceiptWidthMM, $ReceiptHeader,
      $DeliveryFeeMode, $DeliveryFlatFee, $BranchID
    ]);
  } else {
    $q = "INSERT INTO branchsettings
          (BranchID, TaxRate, ServicePercent, Currency, PrintFooter,
           TaxInclusive, RoundingIncrement, PriceSource, AllowNegativeSales,
           DefaultPaymentMethod, ReceiptWidthMM, ReceiptHeader,
           DeliveryFeeMode, DeliveryFlatFee)
          VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
    $pdo->prepare($q)->execute([
      $BranchID, $TaxRate, $ServicePercent, $Currency, $PrintFooter,
      $TaxInclusive, $RoundingIncrement, $PriceSource, $AllowNegativeSales,
      $DefaultPaymentMethod, $ReceiptWidthMM, $ReceiptHeader,
      $DeliveryFeeMode, $DeliveryFlatFee
    ]);
  }

  json_out(['ok'=>true,'msg'=>'تم الحفظ بنجاح']);
}

/* ============== Initial Data ============== */
$user = current_user();
$defaultBranch = (int)($user['branch_id'] ?? 0);

// قائمة الفروع النشطة
$branches = $pdo->query("SELECT BranchID, BranchName FROM branches WHERE IsActive=1 ORDER BY BranchName")
                ->fetchAll(PDO::FETCH_ASSOC);
if (!$defaultBranch && !empty($branches)) {
  $defaultBranch = (int)$branches[0]['BranchID'];
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="utf-8">
  <title>⚙️ إعدادات الفروع</title>
  <meta name="viewport" content="width=device-width, initial-scale=1">
  <!-- Bootstrap RTL + Icons -->
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --brand1:#0f2027; --brand2:#203a43; --brand3:#2c5364;
      --primary:#1abc9c; --primary-600:#16a085; --muted:#6b7280;
      --card:#ffffff; --bg:#f5f6f8; --border:rgba(0,0,0,.08);
    }
    body{background:var(--bg); font-family:"Tajawal", Tahoma, Arial, sans-serif; color:#1f2937}
    header{
      background:linear-gradient(135deg,var(--brand1),var(--brand2),var(--brand3));
      color:#fff; padding:14px 18px; display:flex; align-items:center; justify-content:space-between; gap:12px;
      box-shadow:0 6px 24px rgba(0,0,0,.25); margin-bottom:18px;
    }
    .user-info{display:flex;align-items:center;gap:10px}
    .user-info img{width:38px;height:38px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .panel{background:var(--card); border:1px solid var(--border); border-radius:14px; box-shadow:0 10px 24px rgba(0,0,0,.05);}
    .panel h6{margin:0;font-weight:800}
    .form-control, .form-select{border-radius:10px}
    .label-muted{color:var(--muted); font-size:.9rem}
    .btn-soft{background:#eef2ff; color:#1e40af; border:none}
    .receipt{background:#fff; border:1px dashed #d1d5db; border-radius:10px; padding:16px; font-family: ui-monospace, SFMono-Regular, Menlo, monospace}
    .receipt h5{font-weight:800; margin:0 0 8px}
    .receipt small{color:#6b7280}
    .footer-note{white-space:pre-wrap; line-height:1.6}
    .hint{color:var(--muted)}
    .badge-soft{background:#ecfdf5;color:#065f46;font-weight:700}
  </style>
</head>
<body>
<header class="container-fluid">
  <div class="user-info">
    <img src="uploads/<?= htmlspecialchars($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <div>مرحبًا، <strong><?= htmlspecialchars($_SESSION['admin_username'] ?? 'مستخدم') ?></strong></div>
  </div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="btn btn-light btn-sm"><i class="fa-solid fa-house"></i> الرئيسية</a>
    <a href="logout.php" class="btn btn-outline-light btn-sm"><i class="fa-solid fa-arrow-right-from-bracket"></i> خروج</a>
  </div>
</header>

<div class="container">
  <!-- اختيار الفرع -->
  <div class="panel p-3 mb-3">
    <div class="row g-3 align-items-end">
      <div class="col-md-6">
        <label class="form-label">اختر الفرع</label>
        <select id="branchSel" class="form-select">
          <?php foreach ($branches as $b): ?>
            <option value="<?= (int)$b['BranchID'] ?>" <?= $b['BranchID']===$defaultBranch ? 'selected' : '' ?>>
              <?= htmlspecialchars($b['BranchName']) ?>
            </option>
          <?php endforeach; ?>
        </select>
        <div class="form-text">سيتم حفظ الإعدادات على هذا الفرع فقط.</div>
      </div>
      <div class="col-md-6 text-md-end">
        <span class="badge badge-soft"><i class="fa-solid fa-circle-info"></i> تلميح: النسب تدخل %، والتقريب مثلاً 0.05.</span>
      </div>
    </div>
  </div>

  <div class="row g-3">
    <!-- النموذج -->
    <div class="col-lg-7">
      <div class="panel p-3">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6><i class="fa-solid fa-sliders"></i> إعدادات الفرع</h6>
          <button class="btn btn-outline-secondary btn-sm" id="btnDefaults">
            <i class="fa-solid fa-rotate"></i> استرجاع القيم الافتراضية
          </button>
        </div>

        <form id="settingsForm" class="row g-3" onsubmit="return false;">
          <input type="hidden" id="csrf" value="<?= htmlspecialchars($CSRF) ?>">

          <!-- ضريبة وخدمة وعملة -->
          <div class="col-md-4">
            <label class="form-label">نسبة الضريبة (٪)</label>
            <input type="number" step="0.01" min="0" max="100" class="form-control" id="taxRate" required>
            <div class="form-text label-muted">مثال: 14.00</div>
          </div>
          <div class="col-md-4">
            <label class="form-label">نسبة خدمة الصالة (٪)</label>
            <input type="number" step="0.01" min="0" max="100" class="form-control" id="servicePercent" required>
          </div>
          <div class="col-md-4">
            <label class="form-label">العملة</label>
            <select class="form-select" id="currency" required>
              <option value="EGP">EGP</option><option value="SAR">SAR</option>
              <option value="AED">AED</option><option value="USD">USD</option><option value="EUR">EUR</option>
            </select>
          </div>

          <!-- التسعير والتقريب -->
          <div class="col-md-4">
            <label class="form-label">البيع شامل الضريبة؟</label>
            <select id="taxInclusive" class="form-select">
              <option value="1">نعم (مُوصى به)</option>
              <option value="0">لا</option>
            </select>
          </div>
          <div class="col-md-4">
            <label class="form-label">التقريب إلى</label>
            <input type="number" step="0.01" min="0" class="form-control" id="roundingIncrement" value="0.05">
            <div class="form-text label-muted">مثال: 0.05 (خمسة قروش)</div>
          </div>
          <div class="col-md-4">
            <label class="form-label">مصدر السعر</label>
            <select id="priceSource" class="form-select">
              <option value="branch">سعر الفرع</option>
              <option value="global">سعر عام</option>
            </select>
          </div>

          <!-- POS والمبيعات -->
          <div class="col-md-6">
            <label class="form-label">السماح بسالب المخزون</label>
            <select id="allowNegative" class="form-select">
              <option value="1">مسموح مع تحذير</option>
              <option value="0">غير مسموح</option>
            </select>
          </div>
          <div class="col-md-6">
            <label class="form-label">طريقة الدفع الافتراضية</label>
            <select id="defaultPay" class="form-select">
              <option value="cash">نقدًا</option>
              <option value="card">بطاقة</option>
              <option value="wallet">محفظة</option>
              <option value="split">مقسّم</option>
            </select>
          </div>

          <!-- الطباعة -->
          <div class="col-md-4">
            <label class="form-label">عرض الإيصال (مم)</label>
            <input type="number" min="58" class="form-control" id="receiptWidth" value="80">
          </div>
          <div class="col-12">
            <label class="form-label">عنوان الإيصال (Header)</label>
            <textarea id="printHeader" rows="3" class="form-control" placeholder="اسم المتجر • العنوان • الهاتف"></textarea>
          </div>
          <div class="col-12">
            <label class="form-label">تذييل الطباعة (Footer)</label>
            <textarea id="printFooter" rows="5" class="form-control" placeholder="شكراً لزيارتكم 🌟&#10;الأسعار شاملة الضريبة..."></textarea>
          </div>

          <!-- التوصيل -->
          <div class="col-md-6">
            <label class="form-label">نمط رسوم التوصيل</label>
            <select id="delMode" class="form-select">
              <option value="by_zone">حسب المنطقة</option>
              <option value="flat">قيمة ثابتة</option>
              <option value="none">بدون توصيل</option>
            </select>
          </div>
          <div class="col-md-6">
            <label class="form-label">قيمة ثابتة (لو اختيرت ثابت)</label>
            <input type="number" step="0.01" min="0" class="form-control" id="delFlat" value="0">
          </div>

          <div class="col-12 d-flex gap-2">
            <button id="btnSave" class="btn btn-success">
              <i class="fa-solid fa-floppy-disk"></i> حفظ الإعدادات
            </button>
            <button id="btnPreview" class="btn btn-soft">
              <i class="fa-solid fa-receipt"></i> معاينة الإيصال
            </button>
            <button class="btn btn-outline-secondary" id="btnDefaults">
              <i class="fa-solid fa-rotate"></i> افتراضي
            </button>
          </div>
        </form>

        <div id="alertBox" class="alert mt-3 d-none" role="alert"></div>
      </div>
    </div>

    <!-- المعاينة -->
    <div class="col-lg-5">
      <div class="panel p-3">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h6><i class="fa-solid fa-eye"></i> معاينة الإيصال</h6>
          <small class="hint">عرض تقريبي للطباعة</small>
        </div>
        <div class="receipt" id="preview">
          <h5 id="p_header">—</h5>
          <small id="p_meta">ضريبة: 0% • خدمة: 0% • عملة: EGP • عرض: 80مم</small>
          <hr>
          <div class="footer-note" id="p_footer">—</div>
        </div>
      </div>
    </div>
  </div>
</div>

<!-- JS -->
<script src="https://code.jquery.com/jquery-3.7.0.min.js"></script>
<script>
  const csrf = document.getElementById('csrf').value;
  const $branchSel = $('#branchSel');

  const $tax = $('#taxRate');
  const $svc = $('#servicePercent');
  const $cur = $('#currency');
  const $ti  = $('#taxInclusive');
  const $rnd = $('#roundingIncrement');
  const $ps  = $('#priceSource');
  const $neg = $('#allowNegative');
  const $dpm = $('#defaultPay');
  const $rwd = $('#receiptWidth');
  const $hdr = $('#printHeader');
  const $ftr = $('#printFooter');
  const $dm  = $('#delMode');
  const $df  = $('#delFlat');

  const $alert = $('#alertBox');

  function showAlert(type, msg){
    $alert.removeClass('d-none alert-success alert-danger alert-warning')
          .addClass('alert-' + type).text(msg);
    setTimeout(()=> $alert.addClass('d-none'), 3500);
  }

  function updatePreview(tax, svc, currency, width, headerText, footerText){
    $('#p_header').text(headerText && headerText.trim() ? headerText : 'عنوان الإيصال');
    $('#p_meta').text(`ضريبة: ${(+tax||0)}% • خدمة: ${(+svc||0)}% • عملة: ${currency||'EGP'} • عرض: ${parseInt(width||80,10)}مم`);
    $('#p_footer').text(footerText && footerText.trim() ? footerText : '—');
  }

  function loadSettings(){
    const bid = $branchSel.val();
    if (!bid) return;
    $.getJSON('?action=get&branch_id=' + encodeURIComponent(bid), (resp)=>{
      if (!resp.ok) { showAlert('danger', resp.msg || 'تعذر جلب الإعدادات'); return; }
      const d = resp.data;
      $tax.val((+d.TaxRate).toFixed(2));
      $svc.val((+d.ServicePercent).toFixed(2));
      $cur.val(d.Currency || 'EGP');

      $ti.val((+d.TaxInclusive ? '1':'0'));
      $rnd.val((+d.RoundingIncrement || 0.05).toFixed(2));
      $ps.val(d.PriceSource || 'branch');
      $neg.val((+d.AllowNegativeSales ? '1':'0'));
      $dpm.val(d.DefaultPaymentMethod || 'cash');
      $rwd.val(parseInt(d.ReceiptWidthMM || 80,10));
      $hdr.val(d.ReceiptHeader || '');
      $ftr.val(d.PrintFooter || '');
      $dm.val(d.DeliveryFeeMode || 'by_zone');
      $df.val((+d.DeliveryFlatFee || 0).toFixed(2));

      updatePreview(+d.TaxRate, +d.ServicePercent, d.Currency, d.ReceiptWidthMM, d.ReceiptHeader, d.PrintFooter);
    }).fail(()=> showAlert('danger','خطأ في الاتصال'));
  }

  $(function(){ loadSettings(); });

  $branchSel.on('change', loadSettings);

  $('#btnDefaults').on('click', function(){
    $tax.val('14.00'); $svc.val('0.00'); $cur.val('EGP');
    $ti.val('1'); $rnd.val('0.05'); $ps.val('branch');
    $neg.val('1'); $dpm.val('cash'); $rwd.val('80');
    $hdr.val('اسم المتجر • العنوان • الهاتف');
    $ftr.val('شكراً لزيارتكم 🌟\nالأسعار شاملة الضريبة.\nللاستفسار: 0000000000');
    $dm.val('by_zone'); $df.val('0.00');
    updatePreview(14, 0, 'EGP', 80, $hdr.val(), $ftr.val());
  });

  $('#btnPreview').on('click', function(){
    updatePreview(
      parseFloat($tax.val()||0),
      parseFloat($svc.val()||0),
      $cur.val(),
      parseInt($rwd.val()||80,10),
      $hdr.val(),
      $ftr.val()
    );
  });

  $('#btnSave').on('click', async function(){
    const bid = parseInt($branchSel.val()||'0', 10);
    if (!bid) { showAlert('warning','اختر الفرع أولًا'); return; }

    let tax = parseFloat($tax.val()||'0');
    let svc = parseFloat($svc.val()||'0');
    if (isNaN(tax) || tax<0 || tax>100) { showAlert('warning','قيمة الضريبة بين 0 و 100'); return; }
    if (isNaN(svc) || svc<0 || svc>100) { showAlert('warning','قيمة الخدمة بين 0 و 100'); return; }

    const payload = {
      action:'save',
      csrf: csrf,
      BranchID: bid,
      TaxRate: +tax.toFixed(2),
      ServicePercent: +svc.toFixed(2),
      Currency: $cur.val(),
      PrintFooter: $ftr.val(),

      TaxInclusive: $ti.val()==='1' ? 1 : 0,
      RoundingIncrement: parseFloat($rnd.val()||'0.05'),
      PriceSource: $ps.val(),
      AllowNegativeSales: $neg.val()==='1' ? 1 : 0,
      DefaultPaymentMethod: $dpm.val(),
      ReceiptWidthMM: parseInt($rwd.val()||'80',10),
      ReceiptHeader: $hdr.val(),
      DeliveryFeeMode: $dm.val(),
      DeliveryFlatFee: parseFloat($df.val()||'0')
    };

    try{
      const res = await fetch('settings.php', {
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body: JSON.stringify(payload)
      });
      const data = await res.json();
      if (!data.ok) { showAlert('danger', data.msg || 'فشل الحفظ'); return; }

      showAlert('success', 'تم الحفظ بنجاح');
      updatePreview(payload.TaxRate, payload.ServicePercent, payload.Currency, payload.ReceiptWidthMM, payload.ReceiptHeader, payload.PrintFooter);
    }catch(e){
      showAlert('danger','تعذر الاتصال بالخادم');
    }
  });
</script>
</body>
</html>
