<?php
require_once __DIR__ . '/../core/auth.php';
require_login();

$pdo = db();
$success = $error = "";

/* ================= Helpers عامة ================= */
function esc($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function money($n){ return number_format((float)$n, 2); }
function hasColumn(PDO $pdo, string $table, string $col): bool {
  $st = $pdo->prepare("SELECT COUNT(*) FROM information_schema.columns WHERE table_schema = DATABASE() AND table_name=? AND column_name=?");
  $st->execute([$table,$col]); return (int)$st->fetchColumn() > 0;
}
function clean_str(?string $s, int $max=255): ?string{ $s=trim((string)$s); if($s==='') return null; return mb_substr($s,0,$max); }
function round005(float $amount): array {
  // تقريب لأقرب 0.05 (خمس قروش)
  $rounded = round($amount / 0.05) * 0.05;
  $rounded = round($rounded + 1e-9, 2);
  return [$rounded, $rounded - $amount];
}

/* ===== كشف أعمدة مساعدة ===== */
function detectCols(PDO $pdo, string $table, array $idCands, array $nameCands): array {
  $idCol = $idCands[0]; $nmCol = $nameCands[0];
  foreach ($idCands as $c)  if (hasColumn($pdo,$table,$c)) { $idCol=$c; break; }
  foreach ($nameCands as $c)if (hasColumn($pdo,$table,$c)) { $nmCol=$c; break; }
  return [$idCol,$nmCol];
}

/* ===== صلاحيات ===== */
$canEdit          = true;
$canChangePrice   = !empty($_SESSION['can_change_price']); // عدّلها بنظام صلاحياتك
$canOverrideStock = true; // سماح بالسالب (مع تحذير)

/* ===== CSRF ===== */
if (empty($_SESSION['csrf_token'])) $_SESSION['csrf_token'] = bin2hex(random_bytes(16));
$csrf = $_SESSION['csrf_token'];

/* ================= إنشاء الجداول (إن لم توجد) ================= */
$pdo->exec("
CREATE TABLE IF NOT EXISTS sales (
  id INT AUTO_INCREMENT PRIMARY KEY,
  branch_id INT NOT NULL,
  customer_id INT NULL,
  status ENUM('draft','confirmed','paid','cancelled') NOT NULL DEFAULT 'draft',
  sale_date DATE NOT NULL,
  invoice_no VARCHAR(40) NULL,
  currency_code VARCHAR(10) NOT NULL DEFAULT 'EGP',
  subtotal DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  discount_type ENUM('fixed','percent') NOT NULL DEFAULT 'fixed',
  discount_value DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00, -- % (مخزّن للمرجعية، الأسعار شاملة)
  tax_amount DECIMAL(12,2) NOT NULL DEFAULT 0.00, -- الضريبة المستخرجة من السعر الشامل
  service_fee DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  rounding_adjust DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  total DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  paid DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  notes TEXT NULL,
  zone_id INT NULL,
  deleted_at DATETIME NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS sale_items (
  id INT AUTO_INCREMENT PRIMARY KEY,
  sale_id INT NOT NULL,
  variant_id INT NOT NULL,
  description VARCHAR(255) NULL,
  qty INT NOT NULL DEFAULT 1,
  unit_price_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  line_discount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  tax_rate DECIMAL(5,2) NOT NULL DEFAULT 0.00,
  line_total_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  line_tax_extracted DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  CONSTRAINT fk_si_sale FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
$pdo->exec("
CREATE TABLE IF NOT EXISTS sale_payments (
  id INT AUTO_INCREMENT PRIMARY KEY,
  sale_id INT NOT NULL,
  method VARCHAR(40) NOT NULL,
  amount DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  ref_no VARCHAR(80) NULL,
  notes VARCHAR(255) NULL,
  created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
  CONSTRAINT fk_sp_sale FOREIGN KEY (sale_id) REFERENCES sales(id) ON DELETE CASCADE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");
/* جدول أسعار لكل فرع (اختياري) */
$pdo->exec("
CREATE TABLE IF NOT EXISTS variant_branch_prices (
  variant_id INT NOT NULL,
  branch_id INT NOT NULL,
  price_incl DECIMAL(12,2) NOT NULL DEFAULT 0.00,
  PRIMARY KEY (variant_id, branch_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
");

/* ================= قراءات أساسية ================= */
[$BR_ID_COL, $BR_NAME_COL] = detectCols($pdo,'branches',['id','branch_id','BranchID'],['name','branch_name','BranchName']);
$branches = [];
try {
  $branches = $pdo->query("SELECT `$BR_ID_COL` AS id, `$BR_NAME_COL` AS name FROM branches ORDER BY `$BR_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
} catch(Throwable $e){}

[$CUS_ID_COL,$CUS_NAME_COL] = detectCols($pdo,'customers',['id','CustomerID'],['name','CustomerName','full_name']);
$customers=[];
try{
  $customers = $pdo->query("SELECT `$CUS_ID_COL` AS id, `$CUS_NAME_COL` AS name FROM customers ORDER BY `$CUS_NAME_COL`")->fetchAll(PDO::FETCH_KEY_PAIR);
}catch(Throwable $e){}

/* مناطق (رسوم توصيل) */
[$Z_ID,$Z_NAME] = detectCols($pdo,'zones',['id','ZoneID'],['name','ZoneName','zone']);
$ZONE_FEE_COL = hasColumn($pdo,'zones','delivery_fee') ? 'delivery_fee' : (hasColumn($pdo,'zones','DeliveryFee') ? 'DeliveryFee' : null);
$zones=[];
try{
  $sqlZ = "SELECT `$Z_ID` AS id, `$Z_NAME` AS name".($ZONE_FEE_COL? ", `$ZONE_FEE_COL` AS fee":"")." FROM zones ORDER BY `$Z_NAME`";
  $zones = $pdo->query($sqlZ)->fetchAll(PDO::FETCH_ASSOC);
}catch(Throwable $e){}

/* إعدادات الضريبة من settings */
$settings = ['vat_rate'=>14.00];
try{
  $row = $pdo->query("SELECT * FROM settings LIMIT 1")->fetch(PDO::FETCH_ASSOC);
  if ($row){
    foreach (['vat_rate','VAT','vat','tax_rate','TaxRate'] as $k){
      if (isset($row[$k])) { $settings['vat_rate'] = (float)$row[$k]; break; }
    }
  }
}catch(Throwable $e){}

/* سعر المتغير لفرع: من variant_branch_prices وإلا من product_variants */
function get_variant_price(PDO $pdo, int $variant_id, int $branch_id): float {
  $st=$pdo->prepare("SELECT price_incl FROM variant_branch_prices WHERE variant_id=? AND branch_id=?");
  $st->execute([$variant_id,$branch_id]); $p=$st->fetchColumn();
  if ($p!==false) return (float)$p;
  foreach (['price_incl','price','sale_price','Price'] as $c){
    if (hasColumn($pdo,'product_variants',$c)){
      $st=$pdo->prepare("SELECT `$c` FROM product_variants WHERE id=?");
      $st->execute([$variant_id]); $x=$st->fetchColumn();
      if ($x!==false) return (float)$x;
    }
  }
  return 0.00;
}
/* آخر تكلفة شراء لدفتر الحركة */
function last_purchase_cost(PDO $pdo, int $variantId): float {
  try {
    $st=$pdo->prepare("SELECT unit_cost FROM purchase_items WHERE variant_id=? ORDER BY id DESC LIMIT 1");
    $st->execute([$variantId]); $x=$st->fetchColumn();
    return $x!==false ? (float)$x : 0.0;
  } catch(Throwable $e){ return 0.0; }
}

/* ================= Endpoints Ajax ================= */

/* 1) تحميل أمر واحد */
function load_sale(PDO $pdo, int $id){
  $h = $pdo->prepare("SELECT * FROM sales WHERE id=?");
  $h->execute([$id]); $header=$h->fetch(PDO::FETCH_ASSOC);
  if (!$header) return [null,[]];
  $i = $pdo->prepare("
    SELECT si.*, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS product_name
    FROM sale_items si
    JOIN product_variants v ON v.id=si.variant_id
    JOIN products p ON p.id=v.product_id
    WHERE si.sale_id=?
    ORDER BY si.id
  ");
  $i->execute([$id]); $items = $i->fetchAll(PDO::FETCH_ASSOC);
  return [$header,$items];
}
if (isset($_GET['getSale'])) {
  header('Content-Type: application/json; charset=utf-8');
  $id=(int)$_GET['getSale'];
  [$h,$it]=load_sale($pdo,$id);
  echo json_encode(['header'=>$h,'items'=>$it,'vat_rate'=> (float)$GLOBALS['settings']['vat_rate']], JSON_UNESCAPED_UNICODE); exit;
}

/* 2) بحث المتغيرات (باركود/اسم/SKU) */
if (isset($_GET['searchVariants'])) {
  header('Content-Type: application/json; charset=utf-8');
  $q = trim($_GET['q'] ?? '');
  if ($q===''){ echo json_encode(['ok'=>true,'items'=>[]]); exit; }
  $st=$pdo->prepare("
    SELECT v.id, v.sku, CONCAT(p.name_ar,' | ',IFNULL(p.name_en,'-')) AS label
    FROM product_variants v
    JOIN products p ON p.id=v.product_id
    WHERE v.deleted_at IS NULL
      AND (v.sku LIKE ? OR p.name_ar LIKE ? OR IFNULL(p.name_en,'') LIKE ?)
    ORDER BY p.name_ar, v.id
    LIMIT 20
  ");
  $like="%$q%"; $st->execute([$like,$like,$like]);
  $rows=$st->fetchAll(PDO::FETCH_ASSOC);
  echo json_encode(['ok'=>true,'items'=>$rows], JSON_UNESCAPED_UNICODE); exit;
}

/* 3) رصيد/سعر متغير لفرع */
if (isset($_GET['variantInfo'])) {
  header('Content-Type: application/json; charset=utf-8');
  $vid = (int)($_GET['variant_id'] ?? 0);
  $bid = (int)($_GET['branch_id'] ?? 0);
  if ($vid<=0 || $bid<=0){ echo json_encode(['ok'=>false]); exit; }
  $st=$pdo->prepare("SELECT COALESCE(quantity,0) FROM product_variant_stock WHERE variant_id=? AND branch_id=?");
  $st->execute([$vid,$bid]); $qty=(int)($st->fetchColumn() ?: 0);
  $price = get_variant_price($pdo,$vid,$bid);
  echo json_encode(['ok'=>true,'stock'=>$qty,'price_incl'=>$price, 'vat_rate'=>(float)$settings['vat_rate']]); exit;
}

/* ================= تطبيق المخزون عند البيع ================= */
function apply_sale_stock(PDO $pdo, int $sale_id, int $branch_id){
  // يخصم الكميات (سالب) مع السماح بالسالب
  $st = $pdo->prepare("SELECT variant_id, qty FROM sale_items WHERE sale_id=? ORDER BY id");
  $st->execute([$sale_id]);
  $items=$st->fetchAll(PDO::FETCH_ASSOC);
  foreach ($items as $it){
    $vid=(int)$it['variant_id']; $q=(int)$it['qty'];
    // خصم من المخزون
    $pdo->prepare("
      INSERT INTO product_variant_stock (variant_id, branch_id, quantity, min_quantity)
      VALUES (?,?,?,0)
      ON DUPLICATE KEY UPDATE quantity = quantity - VALUES(quantity)
    ")->execute([$vid,$branch_id,$q]);

    // دفتر حركة
    $uc = last_purchase_cost($pdo,$vid);
    $pdo->prepare("INSERT INTO inventory_movements (variant_id, branch_id, qty_change, unit_cost, ref_type, ref_id)
                   VALUES (?,?,?,?, 'sale', ?)")->execute([$vid,$branch_id,-$q,$uc,$sale_id]);
  }
}

/* ================= الحسابات ================= */
function calc_sale_totals(array $items, string $discount_type, float $discount_value, float $service_fee, float $vat_rate): array {
  // كل الأسعار شاملة الضريبة. نستخرج الضريبة للعرض/التخزين.
  $subtotal_incl = 0.0;
  $tax_extracted = 0.0;
  foreach ($items as $it){
    $qty = max(0,(int)$it['qty']);
    $unit_incl = max(0.0,(float)$it['unit_price_incl']);
    $ldisc = max(0.0,(float)$it['line_discount']);
    $line_incl = max(0.0, ($qty * $unit_incl) - $ldisc);
    $subtotal_incl += $line_incl;

    $rate = isset($it['tax_rate']) ? (float)$it['tax_rate'] : $vat_rate;
    $net  = $rate>0 ? ($line_incl / (1 + $rate/100)) : $line_incl;
    $tax_extracted += ($line_incl - $net);
  }

  $hdrDisc = ($discount_type==='percent') ? ($subtotal_incl * ($discount_value/100)) : $discount_value;
  $hdrDisc = max(0.0, $hdrDisc);
  $after_disc_incl = max(0.0, $subtotal_incl - $hdrDisc);

  // بعد الخصم نضيف خدمة (لا تحتوي ضريبة حسب طلبك؟ إن أردتها خاضعة للضريبة بدّل الحساب)
  $gross = $after_disc_incl + $service_fee;

  // تقريب 0.05 على الإجمالي النهائي
  [$rounded, $adj] = round005($gross);

  // الضريبة النهائية (مُستخرجة) تتناسب مع ما قبل الخدمة (لو الخدمة غير خاضعة)
  // لو الخدمة خاضعة للضريبة، بدّل إلى: $tax_extracted += $service_fee - $service_fee/(1+$vat_rate/100);
  return [
    'subtotal_incl' => $subtotal_incl,
    'tax_extracted' => $tax_extracted,
    'header_discount'=> $hdrDisc,
    'service_fee'   => $service_fee,
    'rounding_adjust'=> $adj,
    'total'         => $rounded
  ];
}

/* ================= POST Actions ================= */
if ($_SERVER['REQUEST_METHOD']==='POST') {
  if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf'] ?? '')) {
    $error = 'طلب غير صالح (CSRF).';
  } elseif (!$canEdit) {
    $error = 'لا تملك صلاحية.';
  } else {
    $action = $_POST['action'] ?? '';
    try {
      if ($action==='save_sale'){
        $id          = ($_POST['id'] ?? '')!=='' ? (int)$_POST['id'] : null;
        $branch_id   = (int)($_POST['branch_id'] ?? 0);
        $customer_id = ($_POST['customer_id'] ?? '')!=='' ? (int)$_POST['customer_id'] : null;
        $status      = $_POST['status'] ?? 'draft';
        $sale_date   = $_POST['sale_date'] ?: date('Y-m-d');
        $currency    = clean_str($_POST['currency_code'] ?? 'EGP', 10);
        $discount_type = $_POST['discount_type'] ?? 'fixed';
        $discount_value= (float)($_POST['discount_value'] ?? 0);
        $vat_rate    = (float)($_POST['tax_rate'] ?? $settings['vat_rate']);
        $service_fee = (float)($_POST['service_fee'] ?? 0);
        $zone_id     = ($_POST['zone_id'] ?? '')!=='' ? (int)$_POST['zone_id'] : null;
        $notes       = clean_str($_POST['notes'] ?? '', 10000);

        if ($branch_id<=0) throw new RuntimeException('اختر فرعًا.');

        // بنود
        $var_ids = array_map('intval', $_POST['item_variant_id'] ?? []);
        $desc_arr= $_POST['item_desc'] ?? [];
        $qty_arr = array_map('intval', $_POST['item_qty'] ?? []);
        $price_arr = array_map('floatval', $_POST['item_unit_price'] ?? []);
        $disc_arr  = array_map('floatval', $_POST['item_line_discount'] ?? []);
        $tax_arr   = array_map('floatval', $_POST['item_tax_rate'] ?? []);

        $items=[];
        for($i=0;$i<count($var_ids);$i++){
          if ($var_ids[$i] <= 0) continue;
          $qty  = max(0,(int)($qty_arr[$i] ?? 0));
          $unit = max(0.0,(float)($price_arr[$i] ?? 0));
          $ld   = max(0.0,(float)($disc_arr[$i] ?? 0));
          $tr   = max(0.0,(float)($tax_arr[$i] ?? $vat_rate));
          $desc = clean_str($desc_arr[$i] ?? '', 255);
          if ($qty===0) continue;
          if (!$canChangePrice){
            // قفل السعر لو مفيش صلاحية: رجّع السعر الرسمي للفرع
            $unit = get_variant_price($pdo, (int)$var_ids[$i], $branch_id);
          }
          $items[] = [
            'variant_id'=>$var_ids[$i], 'description'=>$desc, 'qty'=>$qty,
            'unit_price_incl'=>$unit, 'line_discount'=>$ld, 'tax_rate'=>$tr
          ];
        }
        if (!$items) throw new RuntimeException('أضف بندًا واحدًا على الأقل.');

        // رسوم التوصيل من المنطقة لو محددة
        if ($zone_id && $ZONE_FEE_COL){
          $st=$pdo->prepare("SELECT `$ZONE_FEE_COL` FROM zones WHERE `$Z_ID`=?");
          $st->execute([$zone_id]); $feeRow=$st->fetchColumn();
          if ($feeRow!==false) $service_fee = (float)$feeRow;
        }

        // حساب الإجماليات
        $tot = calc_sale_totals($items, $discount_type, $discount_value, $service_fee, $vat_rate);

        // تحذير سالب؟ جهّز رسالة فقط (لا إيقاف)
        $negativeWarnings = [];
        foreach ($items as $it){
          $st=$pdo->prepare("SELECT COALESCE(quantity,0) FROM product_variant_stock WHERE variant_id=? AND branch_id=?");
          $st->execute([(int)$it['variant_id'],$branch_id]);
          $cur=(int)($st->fetchColumn() ?: 0);
          if ($cur - (int)$it['qty'] < 0){
            $negativeWarnings[] = "تحذير: المتغير #".$it['variant_id']." سيُصبح سالب.";
          }
        }

        $pdo->beginTransaction();
        try{
          if ($id){
            $sql="UPDATE sales SET branch_id=:b, customer_id=:c, status=:st, sale_date=:dt, currency_code=:cc,
                 subtotal=:sub, discount_type=:dtype, discount_value=:dval, tax_rate=:tr, tax_amount=:tax,
                 service_fee=:srv, rounding_adjust=:adj, total=:tot, notes=:notes, zone_id=:z
                 WHERE id=:id AND deleted_at IS NULL";
            $st=$pdo->prepare($sql);
            $st->execute([
              ':b'=>$branch_id, ':c'=>$customer_id, ':st'=>$status, ':dt'=>$sale_date, ':cc'=>$currency,
              ':sub'=>$tot['subtotal_incl'], ':dtype'=>$discount_type, ':dval'=>$discount_value,
              ':tr'=>$vat_rate, ':tax'=>$tot['tax_extracted'], ':srv'=>$tot['service_fee'],
              ':adj'=>$tot['rounding_adjust'], ':tot'=>$tot['total'], ':notes'=>$notes, ':z'=>$zone_id, ':id'=>$id
            ]);
            $pdo->prepare("DELETE FROM sale_items WHERE sale_id=?")->execute([$id]);
            $ins=$pdo->prepare("INSERT INTO sale_items (sale_id, variant_id, description, qty, unit_price_incl, line_discount, tax_rate, line_total_incl, line_tax_extracted)
                                VALUES (?,?,?,?,?,?,?,?,?)");
            foreach($items as $it){
              $qty=$it['qty']; $unit=$it['unit_price_incl']; $ld=$it['line_discount']; $tr=$it['tax_rate'];
              $line_incl = max(0.0, ($qty*$unit)-$ld);
              $net = $tr>0 ? ($line_incl/(1+$tr/100)) : $line_incl;
              $tax = $line_incl - $net;
              $ins->execute([$id,$it['variant_id'],$it['description'],$qty,$unit,$ld,$tr,$line_incl,$tax]);
            }
            if (in_array($status,['confirmed','paid'],true)){
              apply_sale_stock($pdo,$id,$branch_id);
            }
            $pdo->commit();
            $success = '✔️ تم تحديث أمر المبيعات'.( $negativeWarnings? ('<br>'.esc(implode(' / ',$negativeWarnings))) : '' );

          } else {
            $st=$pdo->prepare("INSERT INTO sales (branch_id, customer_id, status, sale_date, currency_code, subtotal, discount_type, discount_value, tax_rate, tax_amount, service_fee, rounding_adjust, total, notes, zone_id)
                               VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)");
            $st->execute([$branch_id,$customer_id,$status,$sale_date,$currency,$tot['subtotal_incl'],$discount_type,$discount_value,$vat_rate,$tot['tax_extracted'],$tot['service_fee'],$tot['rounding_adjust'],$tot['total'],$notes,$zone_id]);
            $sid = (int)$pdo->lastInsertId();
            $ins=$pdo->prepare("INSERT INTO sale_items (sale_id, variant_id, description, qty, unit_price_incl, line_discount, tax_rate, line_total_incl, line_tax_extracted)
                                VALUES (?,?,?,?,?,?,?,?,?)");
            foreach($items as $it){
              $qty=$it['qty']; $unit=$it['unit_price_incl']; $ld=$it['line_discount']; $tr=$it['tax_rate'];
              $line_incl = max(0.0, ($qty*$unit)-$ld);
              $net = $tr>0 ? ($line_incl/(1+$tr/100)) : $line_incl;
              $tax = $line_incl - $net;
              $ins->execute([$sid,$it['variant_id'],$it['description'],$qty,$unit,$ld,$tr,$line_incl,$tax]);
            }
            if (in_array($status,['confirmed','paid'],true)){
              apply_sale_stock($pdo,$sid,$branch_id);
            }
            $pdo->commit();
            $success = '✔️ تم إنشاء أمر المبيعات'.( $negativeWarnings? ('<br>'.esc(implode(' / ',$negativeWarnings))) : '' );
          }

        } catch(Throwable $e){
          $pdo->rollBack();
          throw $e;
        }

      } elseif ($action==='add_payment'){
        $id = (int)($_POST['id'] ?? 0);
        $amount = (float)($_POST['amount'] ?? 0);
        $method = clean_str($_POST['method'] ?? '', 40);
        $ref_no = clean_str($_POST['ref_no'] ?? '', 80);
        $notes  = clean_str($_POST['pay_notes'] ?? '', 255);
        if ($id<=0 || $amount<=0) throw new RuntimeException('مبلغ غير صالح.');
        $pdo->prepare("INSERT INTO sale_payments (sale_id, method, amount, ref_no, notes) VALUES (?,?,?,?,?)")
            ->execute([$id,$method,$amount,$ref_no,$notes]);
        $sum = $pdo->prepare("SELECT COALESCE(SUM(amount),0) FROM sale_payments WHERE sale_id=?");
        $sum->execute([$id]); $paid = (float)$sum->fetchColumn();
        $pdo->prepare("UPDATE sales SET paid=? , status = CASE WHEN ? >= total THEN 'paid' ELSE status END WHERE id=?")
            ->execute([$paid,$paid,$id]);
        $success = '✔️ تم تسجيل الدفعة';

      } elseif ($action==='soft_delete'){
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE sales SET deleted_at=NOW() WHERE id=? AND deleted_at IS NULL")->execute([$id]);
        $success='تم أرشفة الأمر';

      } elseif ($action==='restore'){
        $id=(int)($_POST['id'] ?? 0);
        $pdo->prepare("UPDATE sales SET deleted_at=NULL WHERE id=?")->execute([$id]);
        $success='تم استعادة الأمر';
      }

    } catch(Throwable $e){
      $error = 'خطأ: '.esc($e->getMessage());
    }
  }
}

/* ================= فلاتر قائمة الأوامر ================= */
$q            = trim($_GET['q'] ?? '');
$fBranch      = ($_GET['branch_id'] ?? '')!=='' ? (int)$_GET['branch_id'] : null;
$fStatus      = $_GET['status'] ?? '';
$showArchived = isset($_GET['show_archived']);

$where=[]; $params=[];
$where[] = $showArchived ? "s.deleted_at IS NOT NULL OR s.deleted_at IS NULL" : "s.deleted_at IS NULL";
if ($q!==''){ $where[]="(s.id LIKE :q OR c.`$CUS_NAME_COL` LIKE :q)"; $params[':q']="%$q%"; }
if ($fBranch){ $where[]="s.branch_id=:bid"; $params[':bid']=$fBranch; }
if ($fStatus!==''){ $where[]="s.status=:st"; $params[':st']=$fStatus; }
$whereSql = $where ? ('WHERE '.implode(' AND ', $where)) : '';

$sql = "
  SELECT s.*, b.`$BR_NAME_COL` AS BranchName, c.`$CUS_NAME_COL` AS CustomerName,
         (s.total - s.paid) AS due
  FROM sales s
  LEFT JOIN branches b ON b.`$BR_ID_COL`=s.branch_id
  LEFT JOIN customers c ON c.`$CUS_ID_COL`=s.customer_id
  $whereSql
  ORDER BY s.id DESC
  LIMIT 500";
$st = $pdo->prepare($sql);
$st->execute($params);
$rows = $st->fetchAll(PDO::FETCH_ASSOC);

/* ================= طباعة حراري 80مم ================= */
if (isset($_GET['print']) && $_GET['print']!=='') {
  $id=(int)$_GET['print'];
  [$h,$it]=load_sale($pdo,$id);
  if (!$h){ http_response_code(404); echo "Not found"; exit; }
  // بيانات المتجر/الفرع
  $store = $pdo->query("SELECT * FROM storeinfo LIMIT 1")->fetch(PDO::FETCH_ASSOC) ?: [];
  $branchName = '';
  try{ $bn=$pdo->prepare("SELECT `$BR_NAME_COL` FROM branches WHERE `$BR_ID_COL`=?"); $bn->execute([(int)$h['branch_id']]); $branchName=$bn->fetchColumn(); }catch(Throwable $e){}

  ?><!doctype html>
  <html lang="ar" dir="rtl"><head><meta charset="utf-8"><title>طباعة فاتورة #<?= (int)$id ?></title>
  <style>
    @media print { @page{ size: 80mm auto; margin: 5mm; } }
    body{font-family:"Tajawal",Tahoma,Arial,sans-serif; width:72mm; margin:0 auto; color:#000}
    .c{text-align:center} .r{text-align:right}
    .row{display:flex;justify-content:space-between}
    hr{border:0;border-top:1px dashed #000;margin:6px 0}
    table{width:100%;border-collapse:collapse} th,td{padding:2px 0;font-size:12px}
    th{border-bottom:1px solid #000}
  </style></head><body>
  <div class="c">
    <div style="font-size:16px;font-weight:700"><?= esc($store['store_name'] ?? $store['name'] ?? 'المتجر') ?></div>
    <div><?= esc($branchName ?: '-') ?></div>
    <div style="font-size:11px"><?= esc($store['address'] ?? '-') ?></div>
    <div style="font-size:11px">ض.ق: <?= esc($store['vat_number'] ?? $store['tax_id'] ?? '-') ?></div>
  </div>
  <hr>
  <div style="font-size:12px">فاتورة #: <b><?= (int)$id ?></b> — التاريخ: <?= esc($h['sale_date']) ?></div>
  <table>
    <thead><tr><th class="r">الصنف</th><th>كمية</th><th>سعر</th><th>الإجمالي</th></tr></thead>
    <tbody>
    <?php foreach ($it as $row): ?>
      <tr>
        <td class="r"><?= esc(($row['product_name'] ?? '').' | '.($row['sku'] ?? '')) ?></td>
        <td class="c"><?= (int)$row['qty'] ?></td>
        <td class="c"><?= money($row['unit_price_incl']) ?></td>
        <td class="c"><?= money($row['line_total_incl']) ?></td>
      </tr>
    <?php endforeach; ?>
    </tbody>
  </table>
  <hr>
  <div class="row"><div>الإجمالي قبل الخصم:</div><div><?= money($h['subtotal']) ?></div></div>
  <div class="row"><div>خصم:</div><div><?= money($h['discount_value']) ?> <?= $h['discount_type']==='percent'?'%':'' ?></div></div>
  <div class="row"><div>خدمة/توصيل:</div><div><?= money($h['service_fee']) ?></div></div>
  <div class="row"><div>تقريب:</div><div><?= money($h['rounding_adjust']) ?></div></div>
  <div class="row" style="font-weight:700"><div>الإجمالي:</div><div><?= money($h['total']) ?> <?= esc($h['currency_code']) ?></div></div>
  <div class="row"><div>مدفوع:</div><div><?= money($h['paid']) ?></div></div>
  <div class="row"><div>متبقي:</div><div><?= money((float)$h['total'] - (float)$h['paid']) ?></div></div>
  <hr>
  <div class="c" style="font-size:11px">شكرًا لزيارتكم 🌟</div>
  <script>window.print()</script>
  </body></html><?php
  exit;
}
?>
<!doctype html>
<html lang="ar" dir="rtl">
<head>
  <meta charset="UTF-8">
  <title>أوامر المبيعات</title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.rtl.min.css" rel="stylesheet">
  <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css" rel="stylesheet">
  <style>
    :root{
      --bg:#f7fafc; --text:#1f2937; --muted:#6b7280; --card:#fff; --border:rgba(0,0,0,.06);
      --p1:#c9e4de; --p2:#f6eac2; --p3:#f7d9e3; --p4:#d7e3fc;
      --primary:#7cc5b5; --primary-600:#5fb3a0;
    }
    body.dark{ --bg:#0f1115; --text:#e5e7eb; --muted:#9ca3af; --card:#151922; --border:rgba(255,255,255,.08);
      --p1:#274; --p2:#543; --p3:#345; --p4:#236; --primary:#4fb79f; --primary-600:#3f9884; }
    body{background:var(--bg);color:var(--text);font-family:"Tajawal",Tahoma,Arial,sans-serif}
    header{ background:linear-gradient(90deg,var(--p1),var(--p2),var(--p3),var(--p4)); padding:14px 18px; display:flex; align-items:center; justify-content:space-between; box-shadow:0 6px 24px rgba(0,0,0,.15); }
    .user-info{display:flex;align-items:center;gap:10px;background:rgba(255,255,255,.6);padding:6px 10px;border-radius:999px}
    .user-info img{width:36px;height:36px;border-radius:50%;object-fit:cover;border:2px solid #fff}
    .logout{background:#e74c3c;color:#fff;border:0;border-radius:999px;padding:6px 10px;font-size:12px}
    .logout:hover{background:#c0392b}
    #darkSwitch{background:var(--primary);color:#fff;border:0;border-radius:999px;padding:8px 12px;font-weight:800}
    #darkSwitch:hover{background:var(--primary-600)}
    .back-btn{background: rgba(255,255,255,.85); color:#1f2937; border:0; border-radius:999px; padding:8px 12px; font-weight:800; text-decoration:none}
    .back-btn:hover{background:#fff; color:#111; text-decoration:none}
    .wrap{padding:18px}
    .card-soft{background:var(--card);border:1px solid var(--border);border-radius:16px;box-shadow:0 8px 20px rgba(0,0,0,.06)}
    .card-head{display:flex;gap:10px;align-items:center;justify-content:space-between;padding:14px 16px;border-bottom:1px solid var(--border)}
    .btn-primary{background:var(--primary);border:none}
    .btn-primary:hover{background:var(--primary-600)}
    .pill{display:inline-block;padding:4px 10px;border-radius:999px;background:rgba(0,0,0,.05);font-weight:700}
    .table>:not(caption)>*>*{background:transparent}
    .num{min-width:90px;text-align:center}
    .badge-muted{background:rgba(0,0,0,.06); color:#333}
    .neg{background:#f8d7da;border:1px solid #f5c2c7;border-radius:999px;padding:2px 8px}
  </style>
</head>
<body>
<header>
  <div class="user-info">
    <img src="uploads/<?= esc($_SESSION['avatar'] ?? 'default.png') ?>" alt="Avatar">
    <span class="fw-bold"><?= esc($_SESSION['admin_username'] ?? 'Admin') ?></span>
    <a class="logout" href="logout.php"><i class="fa-solid fa-right-from-bracket"></i> خروج</a>
  </div>
  <div class="title fw-bold"><i class="fa-solid fa-file-invoice-dollar me-2"></i> أوامر المبيعات</div>
  <div class="d-flex align-items-center gap-2">
    <a href="dashboard.php" class="back-btn"><i class="fa-solid fa-house"></i> لوحة التحكم</a>
    <button id="darkSwitch">🌙</button>
  </div>
</header>

<div class="wrap">
  <!-- فلاتر -->
  <div class="card-soft mb-3">
    <div class="card-head">
      <form class="d-flex flex-wrap gap-2 align-items-center" method="get">
        <input type="text" class="form-control" name="q" style="min-width:240px" placeholder="بحث برقم الأمر/اسم العميل" value="<?= esc($q) ?>">
        <select class="form-select" name="branch_id" style="min-width:200px">
          <option value="">كل الفروع</option>
          <?php foreach ($branches as $bid=>$bn): ?>
            <option value="<?= (int)$bid ?>" <?= $fBranch===$bid?'selected':''; ?>><?= esc($bn) ?></option>
          <?php endforeach; ?>
        </select>
        <select class="form-select" name="status" style="min-width:200px">
          <option value="">كل الحالات</option>
          <?php foreach (['draft'=>'مسودة','confirmed'=>'مؤكد','paid'=>'مدفوع','cancelled'=>'ملغي'] as $k=>$v): ?>
            <option value="<?= $k ?>" <?= $fStatus===$k?'selected':''; ?>><?= $v ?></option>
          <?php endforeach; ?>
        </select>
        <div class="form-check">
          <input class="form-check-input" type="checkbox" id="show_archived" name="show_archived" value="1" <?= $showArchived?'checked':''; ?>>
          <label class="form-check-label" for="show_archived">عرض المؤرشف</label>
        </div>
        <button class="btn btn-primary"><i class="fa-solid fa-magnifying-glass"></i> تطبيق</button>
        <?php if ($q || $fBranch || $fStatus || $showArchived): ?>
          <a href="sales.php" class="btn btn-outline-secondary">مسح الفلاتر</a>
        <?php endif; ?>
      </form>

      <?php if ($canEdit): ?>
      <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#saleModal" onclick="openCreate()">
        <i class="fa-solid fa-plus"></i> أمر جديد
      </button>
      <?php endif; ?>
    </div>
  </div>

  <!-- جدول الأوامر -->
  <div class="card-soft">
    <div class="p-3">
      <?php if ($success): ?><div class="alert alert-success"><?= $success ?></div><?php endif; ?>
      <?php if ($error): ?><div class="alert alert-danger"><?= $error ?></div><?php endif; ?>

      <div class="table-responsive">
        <table class="table align-middle text-center">
          <thead class="table-light">
            <tr>
              <th>#</th>
              <th>التاريخ</th>
              <th>الفرع</th>
              <th>العميل</th>
              <th>الحالة</th>
              <th>الإجمالي</th>
              <th>مدفوع</th>
              <th>متبقي</th>
              <th>إجراءات</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="9" class="text-muted">لا توجد بيانات.</td></tr>
          <?php else: foreach ($rows as $r): $due=(float)$r['due']; ?>
            <tr>
              <td><?= (int)$r['id'] ?></td>
              <td><?= esc($r['sale_date']) ?></td>
              <td><?= esc($r['BranchName'] ?? '-') ?></td>
              <td><?= esc($r['CustomerName'] ?? 'عميل نقدي') ?></td>
              <td>
                <?php
                  $badges = [
                    'draft'=>'<span class="badge bg-secondary">مسودة</span>',
                    'confirmed'=>'<span class="badge bg-info text-dark">مؤكد</span>',
                    'paid'=>'<span class="badge bg-success">مدفوع</span>',
                    'cancelled'=>'<span class="badge bg-dark">ملغي</span>',
                  ];
                  echo $badges[$r['status']] ?? esc($r['status']);
                ?>
              </td>
              <td><?= money($r['total']) ?></td>
              <td><?= money($r['paid']) ?></td>
              <td><?= money($due) ?> <?= $due<0?'<span class="neg">رصيد دائن</span>':'' ?></td>
              <td>
                <div class="btn-group">
                  <a class="btn btn-sm btn-outline-dark" href="?print=<?= (int)$r['id'] ?>" target="_blank" title="طباعة"><i class="fa-solid fa-print"></i></a>
                  <button class="btn btn-sm btn-outline-secondary" onclick="openEdit(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-pen-to-square"></i></button>
                  <button class="btn btn-sm btn-outline-primary" onclick="openPayment(<?= (int)$r['id'] ?>)"><i class="fa-solid fa-money-bill-wave"></i></button>
                  <?php if ($r['deleted_at'] === null): ?>
                    <form method="post" class="d-inline" onsubmit="return confirm('أرشفة الأمر؟')">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="soft_delete">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-danger"><i class="fa-solid fa-box-archive"></i></button>
                    </form>
                  <?php else: ?>
                    <form method="post" class="d-inline">
                      <input type="hidden" name="csrf" value="<?= $csrf ?>">
                      <input type="hidden" name="action" value="restore">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <button class="btn btn-sm btn-outline-success"><i class="fa-solid fa-rotate-left"></i></button>
                    </form>
                  <?php endif; ?>
                </div>
              </td>
            </tr>
          <?php endforeach; endif; ?>
          </tbody>
        </table>
      </div>

    </div>
  </div>
</div>

<!-- Modal: Create/Edit Sale -->
<div class="modal fade" id="saleModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-xl modal-dialog-centered">
    <div class="modal-content" style="border-radius:16px">
      <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
        <h5 class="modal-title"><i class="fa-solid fa-file-invoice me-2"></i> <span id="modalTitle">أمر جديد</span></h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>

      <form method="post" class="modal-body">
        <input type="hidden" name="csrf" value="<?= $csrf ?>">
        <input type="hidden" name="action" value="save_sale">
        <input type="hidden" name="id" id="f_id">

        <div class="row g-3">
          <div class="col-md-3">
            <label class="form-label">الفرع *</label>
            <select class="form-select" name="branch_id" id="f_branch" required>
              <option value="">— اختر —</option>
              <?php foreach ($branches as $bid=>$bn): ?>
                <option value="<?= (int)$bid ?>"><?= esc($bn) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">العميل</label>
            <select class="form-select" name="customer_id" id="f_customer">
              <option value="">عميل نقدي</option>
              <?php foreach ($customers as $cid=>$cname): ?>
                <option value="<?= (int)$cid ?>"><?= esc($cname) ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">الحالة</label>
            <select class="form-select" name="status" id="f_status">
              <option value="draft">مسودة</option>
              <option value="confirmed">مؤكد</option>
              <option value="paid">مدفوع</option>
              <option value="cancelled">ملغي</option>
            </select>
          </div>
          <div class="col-md-3">
            <label class="form-label">تاريخ</label>
            <input type="date" class="form-control" name="sale_date" id="f_date" value="<?= date('Y-m-d') ?>">
          </div>

          <div class="col-md-2">
            <label class="form-label">العملة</label>
            <input type="text" class="form-control" name="currency_code" id="f_currency" value="EGP">
          </div>
          <div class="col-md-3">
            <label class="form-label">ضريبة % (شامل)</label>
            <input type="number" step="0.01" class="form-control" name="tax_rate" id="f_tax_rate" value="<?= (float)$settings['vat_rate'] ?>">
          </div>
          <div class="col-md-3">
            <label class="form-label">خصم عام</label>
            <div class="input-group">
              <select class="form-select" name="discount_type" id="f_discount_type" style="max-width:110px">
                <option value="fixed">قيمة</option>
                <option value="percent">%</option>
              </select>
              <input type="number" step="0.01" class="form-control" name="discount_value" id="f_discount_value" value="0">
            </div>
          </div>
          <div class="col-md-4">
            <label class="form-label">منطقة التوصيل</label>
            <select class="form-select" name="zone_id" id="f_zone">
              <option value="">— بدون —</option>
              <?php foreach ($zones as $z): ?>
                <option value="<?= (int)$z['id'] ?>" data-fee="<?= isset($z['fee'])?(float)$z['fee']:0 ?>"><?= esc($z['name']).(isset($z['fee'])?' — رسوم: '.money($z['fee']).' EGP':'') ?></option>
              <?php endforeach; ?>
            </select>
          </div>
          <div class="col-md-2">
            <label class="form-label">رسوم/خدمة</label>
            <input type="number" step="0.01" class="form-control" name="service_fee" id="f_service_fee" value="0">
          </div>

          <div class="col-12">
            <label class="form-label">ملاحظات</label>
            <textarea class="form-control" name="notes" id="f_notes" rows="2"></textarea>
          </div>

          <!-- بحث وباركود -->
          <div class="col-12">
            <div class="d-flex flex-wrap gap-2 align-items-center">
              <input type="text" id="scanSku" class="form-control" placeholder="اسكان باركود / اكتب SKU أو اسم واضغط Enter" style="min-width:280px">
              <span class="badge badge-muted">الأسعار شاملة الضريبة</span>
              <span class="badge badge-muted">التقريب لأقرب 0.05</span>
            </div>
          </div>

          <!-- بنود -->
          <div class="col-12">
            <div class="table-responsive">
              <table class="table table-sm align-middle text-center">
                <thead class="table-light">
                  <tr>
                    <th style="min-width:260px">المتغيّر</th>
                    <th>الوصف</th>
                    <th class="num">المخزون</th>
                    <th class="num">كمية</th>
                    <th class="num">سعر شامل</th>
                    <th class="num">خصم سطر</th>
                    <th class="num">ضريبة %</th>
                    <th class="num">إجمالي سطر</th>
                    <th></th>
                  </tr>
                </thead>
                <tbody id="itemsBody"></tbody>
              </table>
            </div>
            <button type="button" class="btn btn-outline-primary" onclick="addRow()">➕ إضافة بند</button>
          </div>

          <!-- الإجماليات -->
          <div class="col-12">
            <div class="row g-3 justify-content-end">
              <div class="col-md-6">
                <div class="p-2" style="background:rgba(0,0,0,.03);border-radius:12px">
                  <div class="d-flex justify-content-between"><span>المجموع (شامل):</span><strong id="t_subtotal">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>خصم عام:</span><strong id="t_hdrdisc">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>خدمة/توصيل:</span><strong id="t_srv">0.00</strong></div>
                  <div class="d-flex justify-content-between"><span>تقريب 0.05:</span><strong id="t_round">0.00</strong></div>
                  <hr class="my-2">
                  <div class="d-flex justify-content-between fs-5"><span>الإجمالي:</span><strong id="t_total">0.00</strong></div>
                </div>
              </div>
            </div>
          </div>

          <div class="col-12 text-center">
            <button class="btn btn-primary px-4"><i class="fa-solid fa-floppy-disk"></i> حفظ</button>
          </div>
        </div>
      </form>
    </div>
  </div>
</div>

<!-- Modal: Payment -->
<div class="modal fade" id="paymentModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content" style="border-radius:16px">
    <div class="modal-header" style="background:linear-gradient(90deg,var(--p1),var(--p2));">
      <h5 class="modal-title"><i class="fa-solid fa-money-bill-wave me-2"></i> تسجيل دفعة</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form method="post" class="modal-body">
      <input type="hidden" name="csrf" value="<?= $csrf ?>">
      <input type="hidden" name="action" value="add_payment">
      <input type="hidden" name="id" id="pay_sale_id">
      <div class="mb-2">
        <label class="form-label">المبلغ</label>
        <input type="number" step="0.01" class="form-control" name="amount" required>
      </div>
      <div class="mb-2">
        <label class="form-label">الطريقة</label>
        <input type="text" class="form-control" name="method" placeholder="Cash/Card/Wallet/Transfer">
      </div>
      <div class="mb-2">
        <label class="form-label">مرجع</label>
        <input type="text" class="form-control" name="ref_no">
      </div>
      <div class="mb-2">
        <label class="form-label">ملاحظات</label>
        <input type="text" class="form-control" name="pay_notes">
      </div>
      <div class="text-center">
        <button class="btn btn-primary px-4">حفظ</button>
      </div>
    </form>
  </div></div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<script>
  // Dark
  const body=document.body, darkSwitch=document.getElementById('darkSwitch');
  if (localStorage.getItem('darkMode')==='true'){ body.classList.add('dark'); darkSwitch.textContent='☀️'; }
  darkSwitch?.addEventListener('click', ()=>{ body.classList.toggle('dark'); const d=body.classList.contains('dark'); localStorage.setItem('darkMode', d); darkSwitch.textContent=d?'☀️':'🌙'; });

  const saleModal = new bootstrap.Modal(document.getElementById('saleModal'));
  const paymentModal = new bootstrap.Modal(document.getElementById('paymentModal'));
  const itemsBody = document.getElementById('itemsBody');

  function openCreate(){
    document.getElementById('modalTitle').textContent='أمر جديد';
    ['f_id','f_notes','f_discount_value','f_service_fee'].forEach(id=>{ const el=document.getElementById(id); if(el) el.value = (id==='f_discount_value' || id==='f_service_fee')?'0':''; });
    document.getElementById('f_status').value='draft';
    document.getElementById('f_date').value = new Date().toISOString().slice(0,10);
    document.getElementById('f_currency').value='EGP';
    document.getElementById('f_discount_type').value='fixed';
    document.getElementById('f_zone').value='';
    itemsBody.innerHTML=''; addRow(); calcTotals();
  }

  function openEdit(id){
    fetch(location.href + (location.search ? '&':'?') + 'getSale=' + encodeURIComponent(id))
      .then(r=>r.json())
      .then(data=>{
        if(!data || !data.header){ alert('لم يتم العثور على البيانات'); return; }
        fillForm(data.header, data.items);
        saleModal.show();
      })
      .catch(()=> alert('تعذر التحميل'));
  }

  function fillForm(h, items){
    document.getElementById('modalTitle').textContent='تعديل أمر #' + (h.id||'');
    document.getElementById('f_id').value = h.id||'';
    document.getElementById('f_branch').value = h.branch_id||'';
    document.getElementById('f_customer').value = h.customer_id||'';
    document.getElementById('f_status').value = h.status||'draft';
    document.getElementById('f_date').value = h.sale_date||'';
    document.getElementById('f_currency').value = h.currency_code||'EGP';
    document.getElementById('f_tax_rate').value = h.tax_rate||0;
    document.getElementById('f_discount_type').value = h.discount_type||'fixed';
    document.getElementById('f_discount_value').value = h.discount_value||0;
    document.getElementById('f_service_fee').value = h.service_fee||0;
    document.getElementById('f_zone').value = h.zone_id||'';
    document.getElementById('f_notes').value = h.notes||'';
    itemsBody.innerHTML='';
    (items||[]).forEach(addRowFromItem);
    if ((items||[]).length===0) addRow();
    calcTotals();
  }

  function addRowFromItem(it){
    const tr = document.createElement('tr'); tr.className='item-row';
    const desc = it.description||'';
    tr.innerHTML = `
      <td>
        <input class="form-control" placeholder="اكتب SKU/اسم واضغط Enter" value="${it.variant_id? (it.variant_id+' | '+(it.product_name||'')+(it.sku?(' | '+it.sku):'')) : ''}" onkeydown="variantKey(this, event)">
        <input type="hidden" name="item_variant_id[]" value="${it.variant_id||''}">
      </td>
      <td><input type="text" name="item_desc[]" class="form-control" value="${escapeHtml(desc)}"></td>
      <td class="num"><span class="pill stock-pill">?</span></td>
      <td><input type="number" name="item_qty[]" class="form-control text-center" value="${it.qty||1}" oninput="calcLine(this.closest('tr'))"></td>
      <td><input type="number" step="0.01" name="item_unit_price[]" class="form-control text-center" value="${(it.unit_price_incl||0).toFixed? it.unit_price_incl : (it.unit_price_incl||0)}" oninput="calcLine(this.closest('tr'))" ${<?= $canChangePrice?'false':'true' ?> ? '' : 'readonly' } ></td>
      <td><input type="number" step="0.01" name="item_line_discount[]" class="form-control text-center" value="${it.line_discount||0}" oninput="calcLine(this.closest('tr'))"></td>
      <td><input type="number" step="0.01" name="item_tax_rate[]" class="form-control text-center" value="${it.tax_rate ?? (document.getElementById('f_tax_rate').value||0)}" oninput="calcLine(this.closest('tr'))"></td>
      <td class="num"><span class="pill line-total">0.00</span></td>
      <td><button type="button" class="btn btn-sm btn-outline-danger" onclick="this.closest('tr').remove(); calcTotals();">🗑️</button></td>
    `;
    itemsBody.appendChild(tr);
    // حمّل الرصيد والسعر
    if (it.variant_id) refreshVariantInfo(tr, it.variant_id);
    calcLine(tr);
  }

  function addRow(){ addRowFromItem({variant_id:'', qty:1, unit_price_incl:0, line_discount:0}); }

  async function refreshVariantInfo(tr, vid){
    const bid = document.getElementById('f_branch').value||'';
    if (!bid) return;
    const res = await fetch(location.href + (location.search ? '&':'?') + 'variantInfo=1&variant_id='+encodeURIComponent(vid)+'&branch_id='+encodeURIComponent(bid));
    const d = await res.json();
    if (!d.ok) return;
    const stockPill = tr.querySelector('.stock-pill'); stockPill.textContent = d.stock;
    const priceEl = tr.querySelector('[name="item_unit_price[]"]');
    if (!priceEl.value || parseFloat(priceEl.value)==0){ priceEl.value = (d.price_incl||0).toFixed(2); }
    calcLine(tr);
  }

  function variantKey(inp, e){
    if (e.key !== 'Enter') return;
    e.preventDefault();
    const q = inp.value.trim(); if (!q) return;
    // المتوقع صيغة "id | name | sku" أو كتابة SKU/اسم
    const idFromText = parseInt(q.split('|')[0],10);
    if (!isNaN(idFromText)){ selectVariant(inp, idFromText); return; }
    fetch(location.href + (location.search ? '&':'?') + 'searchVariants=1&q='+encodeURIComponent(q))
      .then(r=>r.json()).then(d=>{
        const it = (d.items||[])[0];
        if (!it){ alert('غير موجود'); return; }
        selectVariant(inp, it.id, `${it.id} | ${it.label} | ${it.sku}`);
      });
  }
  function selectVariant(inp, id, display){
    const tr = inp.closest('tr');
    inp.value = display || (id+'');
    tr.querySelector('[name="item_variant_id[]"]').value = id;
    refreshVariantInfo(tr, id);
  }

  function calcLine(tr){
    const qty = parseFloat(tr.querySelector('[name="item_qty[]"]').value||0);
    const unit= parseFloat(tr.querySelector('[name="item_unit_price[]"]').value||0);
    const disc= parseFloat(tr.querySelector('[name="item_line_discount[]"]').value||0);
    const t   = parseFloat(tr.querySelector('[name="item_tax_rate[]"]').value||0);
    let line = Math.max(0, qty*unit - disc);
    tr.querySelector('.line-total').textContent = line.toFixed(2);

    // تحذير سالب (بناءً على المخزون الحالي)
    const stockTxt = tr.querySelector('.stock-pill').textContent;
    const stock = parseInt(stockTxt||0);
    if (!isNaN(stock) && qty>stock){
      tr.querySelector('.stock-pill').classList.add('neg');
      tr.querySelector('.stock-pill').title = 'سيصبح المخزون سالب عند التأكيد/الدفع';
    } else {
      tr.querySelector('.stock-pill').classList.remove('neg');
      tr.querySelector('.stock-pill').title = '';
    }
    calcTotals();
  }

  function calcTotals(){
    let subtotal=0;
    document.querySelectorAll('#itemsBody .item-row').forEach(tr=>{
      const lt = parseFloat(tr.querySelector('.line-total').textContent||0);
      subtotal += lt;
    });
    const dtype = document.getElementById('f_discount_type').value;
    const dval  = parseFloat(document.getElementById('f_discount_value').value||0);
    const srv   = parseFloat(document.getElementById('f_service_fee').value||0);

    let hdrDisc = (dtype==='percent') ? (subtotal*(dval/100)) : dval; hdrDisc=Math.max(0,hdrDisc);
    const after = Math.max(0, subtotal - hdrDisc) + srv;

    // تقريب 0.05
    const rounded = Math.round((after/0.05))*0.05;
    const adj = rounded - after;

    document.getElementById('t_subtotal').textContent = subtotal.toFixed(2);
    document.getElementById('t_hdrdisc').textContent  = hdrDisc.toFixed(2);
    document.getElementById('t_srv').textContent      = srv.toFixed(2);
    document.getElementById('t_round').textContent    = adj.toFixed(2);
    document.getElementById('t_total').textContent    = rounded.toFixed(2);
  }

  // تحديث رسوم حسب المنطقة
  document.getElementById('f_zone')?.addEventListener('change', (e)=>{
    const opt = e.target.selectedOptions[0];
    const fee = parseFloat(opt?.dataset?.fee || 0);
    if (!isNaN(fee)) { document.getElementById('f_service_fee').value = fee.toFixed(2); calcTotals(); }
  });
  ['f_discount_type','f_discount_value','f_service_fee'].forEach(id=>{
    const el=document.getElementById(id); el && el.addEventListener('input', calcTotals);
  });

  // سكان باركود/بحث سريع: يضيف أو يزيد الكمية
  document.getElementById('scanSku')?.addEventListener('keydown', async (e)=>{
    if (e.key!=='Enter') return;
    e.preventDefault();
    const q = e.target.value.trim(); if (!q) return;
    const res = await fetch(location.href + (location.search ? '&':'?') + 'searchVariants=1&q='+encodeURIComponent(q));
    const data = await res.json();
    const match = (data.items||[]).find(x=> (x.sku||'').toLowerCase()===q.toLowerCase()) || (data.items||[])[0];
    if (!match){ alert('غير موجود'); return; }
    const exist = [...document.querySelectorAll('#itemsBody .item-row')].find(tr=> tr.querySelector('[name="item_variant_id[]"]').value == match.id);
    if (exist){
      const qEl = exist.querySelector('[name="item_qty[]"]'); qEl.value = (parseInt(qEl.value||0)+1); calcLine(exist);
    } else {
      addRowFromItem({variant_id:match.id, qty:1, unit_price_incl:0, line_discount:0});
      const last = itemsBody.lastElementChild;
      last.querySelector('td:first-child input').value = `${match.id} | ${match.label} | ${match.sku}`;
      refreshVariantInfo(last, match.id);
    }
    e.target.value='';
  });

  function openPayment(id){ document.getElementById('pay_sale_id').value = id; paymentModal.show(); }

  function escapeHtml(s){ return (s||'').toString().replace(/[&<>"']/g, m=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;' }[m])); }
</script>
</body>
</html>
